<?php

namespace IZON\Forms;

use Exception;
use InvalidArgumentException;
use IZON\Forms\Fields\BaseField;
use IZON\Forms\Fields\CharField;
use IZON\Forms\Fields\CheckboxField;
use IZON\Forms\Fields\CheckboxListField;
use IZON\Forms\Fields\DateFiled;
use IZON\Forms\Fields\DateTimeFiled;
use IZON\Forms\Fields\EmailField;
use IZON\Forms\Fields\EntityHiddenField;
use IZON\Forms\Fields\FloatField;
use IZON\Forms\Fields\FloatRangeField;
use IZON\Forms\Fields\HiddenField;
use IZON\Forms\Fields\IdField;
use IZON\Forms\Fields\IntegerField;
use IZON\Forms\Fields\IntegerRangeField;
use IZON\Forms\Fields\MultipleEmailField;
use IZON\Forms\Fields\PasswordField;
use IZON\Forms\Fields\PhoneField;
use IZON\Forms\Fields\RadioListField;
use IZON\Forms\Fields\SelectField;
use IZON\Forms\Fields\TextField;
use function IZON\Arrays\isEmpty;

/**
 * Vygenerovani formularoveho pole na zaklade templatu vzhledu
 *
 * @author Vitezslav Jahn <jahn@izon.cz>
 */
class FormFieldTemplate
{
    public const TYPE_BACKEND = 'admin';
    public const TYPE_FRONTEND = 'web';
    public const TYPE_FRONTEND_ADMIN = 'web-admin';
    public const LABEL_TEXT_BEFORE_FIELD = 'before';
    public const LABEL_TEXT_AFTER_FIELD = 'after';
    public const ERROR_BEFORE_INPUT = 'before';
    public const ERROR_AFTER_INPUT = 'after';


    private function __construct()
    {
    }

    /**
     * Return HTML code for form field
     *
     * @param string $templateFile rendering template name
     * @param string $type admin|web|web-admin type of template to use
     * @param BaseField $field
     * @param array $params array of parameters
     * @return string
     */
    public static function getFiled(string $templateFile, string $type, BaseField $field, array $params = []): string
    {
        $template = self::loadTemplate($type, $templateFile);
        $paramsField = $params['field'] ?? ['attributes' => []];

        $fieldClass = get_class($field);
        $templateType = null;
        $fieldType = null;
        switch ($fieldClass) {
            case CharField::class:
                $templateType = 'text';
                $fieldType = 'text';
                break;
            case DateFiled::class:
                $templateType = 'date';
                $fieldType = 'text';
                break;
            case DateTimeFiled::class:
                $templateType = 'datetime';
                $fieldType = 'text';
                break;
            case CheckboxField::class:
                $templateType = 'checkbox';
                $fieldType = 'checkbox';
                break;
            case EmailField::class:
            case MultipleEmailField::class:
                $templateType = 'text';
                $fieldType = 'email';
                break;
            case PhoneField::class:
                $templateType = 'text';
                $fieldType = 'tel';
                break;
            case FloatField::class:
            case IntegerField::class:
            case IntegerRangeField::class:
            case FloatRangeField::class:
                $templateType = 'text';
                $fieldType = 'number';
                break;
            case PasswordField::class:
                $templateType = 'password';
                $fieldType = 'password';
                break;
            case SelectField::class:
                $templateType = 'select';
                $fieldType = 'select';
                break;
            case RadioListField::class:
                $templateType = 'radio';
                $fieldType = 'radio';
                break;
            case CheckboxListField::class:
                $templateType = 'checkboxList';
                $fieldType = 'checkboxList';
                break;
            case TextField::class:
                $templateType = 'textarea';
                $fieldType = 'textarea';
                break;
            case IdField::class:
            case HiddenField::class:
            case EntityHiddenField::class:
                $templateType = 'hidden';
                $fieldType = 'hidden';
                break;
            default:
                throw new InvalidArgumentException('Field ' . $fieldClass . ' is not supported');
        }

        $templateParams = $template[$templateType]['field'] ?? [];
        $fieldParams = $params['field'] ?? [];

        $input = null;
        switch ($fieldType) {
            case 'text':
            case 'email':
            case 'tel':
            case 'number':
            case 'password':
            case 'checkbox':
                $input = self::parseInput(
                    $fieldType,
                    $templateParams,
                    $field,
                    $fieldParams
                );
                break;
            case 'select':
                $input = self::parseSelect($templateParams, $field, $fieldParams); // @phpstan-ignore argument.type
                break;
            case 'radio':
                $input = self::parseRadio($templateParams, $field, $fieldParams);
                break;
            case 'checkboxList':
                $input = self::parseCheckboxListField(
                    $templateParams,
                    $field,  // @phpstan-ignore argument.type
                    $fieldParams
                );
                break;
            case 'textarea':
                $input = self::parseTextArea($templateParams, $field, $fieldParams);
                break;
            case 'hidden':
                $input = self::parseHiddenField($field, $fieldParams);
                break;
        }

        if ($fieldType === 'hidden' || ($params['raw'] ?? false) === true) {
            return $input;
        }

        $fieldAttributes = self::getFieldAttributeArray(
            $templateParams['attributes'] ?? [],
            $field,
            $fieldParams['attributes'] ?? []
        );

        // WRAPPER
        $wrapperSettings = (array)$template[$templateType]['wrapper'];
        if ($field->hasErrors()) {
            if (array_key_exists('attributes', $wrapperSettings)) {
                $wrapperSettings['attributes']['class'] .= ' error';
            } else {
                $wrapperSettings['attributes']['class'] = 'error';
            }
        }
        if (isset($fieldAttributes['readonly']) || isset($fieldAttributes['disabled'])) {
            if (array_key_exists('attributes', $wrapperSettings)) {
                $wrapperSettings['attributes']['class'] .= ' readonly';
            } else {
                $wrapperSettings['attributes']['class'] = 'readonly';
            }
        }
        if (!array_key_exists('captionPosition', $wrapperSettings)) {
            $wrapperSettings['captionPosition'] = self::LABEL_TEXT_BEFORE_FIELD;
        }
        if (
            array_key_exists('captionPosition', $params['wrapper'] ?? [])
            && !empty($params['wrapper']['captionPosition'] ?? null)
        ) {
            $wrapperSettings['captionPosition'] = $params['wrapper']['captionPosition'];
        }
        if (!array_key_exists('errorPosition', $wrapperSettings)) {
            $wrapperSettings['errorPosition'] = self::ERROR_AFTER_INPUT;
        }
        if (
            array_key_exists('errorPosition', $params['wrapper'] ?? [])
            && !empty($params['wrapper']['errorPosition'] ?? null)
        ) {
            $wrapperSettings['errorPosition'] = $params['wrapper']['errorPosition'];
        }
        $wrapper = self::generateWrapper($wrapperSettings, $params['wrapper'] ?? []);


        // CAPTION
        $caption = '';
        if (!($params['noCaption'] ?? false)) {
            $captionText = '';
            if (
                array_key_exists('caption', $params)
                && is_array($params['caption'])
                && array_key_exists('text', $params['caption'])
            ) {
                $captionText = $params['caption']['text'];
            }
            if (empty($captionText)) {
                $captionText = $field->getLabel();
            }
            $caption = self::generateCaption(
                $template[$templateType]['caption'] ?? [],
                $captionText,
                $params['caption'] ?? []
            );
        }

        $error = $field->hasErrors() ? self::generateFieldError(
            $template[$templateType]['error'],
            $field->getErrors(),
            $params['error'] ?? []
        ) : '';

        $content = '';
        switch ($wrapperSettings['errorPosition']) {
            case self::ERROR_BEFORE_INPUT:
                $content = $error . $input;
                break;
            default:
                $content = $input . $error;
                break;
        }
        switch ($wrapperSettings['captionPosition']) {
            case self::LABEL_TEXT_BEFORE_FIELD:
                $content = $caption . $input . $error . "\n";
                break;
            case self::LABEL_TEXT_AFTER_FIELD:
                $content = $input . $error . $caption . "\n";
                break;
        }

        return str_replace('__CONTENT__', $content, $wrapper);
    }

    /**
     *
     * @param string $typeTemplatesDir directory where to search template definitions
     * @param string $file template file name
     * @return array
     * @throws Exception
     */
    protected static function loadTemplate(string $typeTemplatesDir, string $file): array
    {
        // For web form fields definition and for admin form fields definition backward compatibility
        $path = $typeTemplatesDir . '/' . $file . '.json';
        if (file_exists($path)) {
            $formTemplates = file_get_contents($path);
            return json_decode($formTemplates, true);
        }

        throw new Exception('Soubor s definici vzhledu formularovych poli - ' . $path . ' - nebyl nalezen.');
    }

    /**
     *
     * @param string $fieldType [text|checkbox|number|password]
     * @param array $pattern
     * @param BaseField $field
     * @param array $params
     * @return string
     */
    protected static function parseInput(string $fieldType, $pattern, Fields\BaseField $field, $params = [])
    {
        if (isset($pattern['type'])) {
            $fieldType = $pattern['type'];
        }

        $attrs = $pattern['attributes'] ?? [];
        if (!in_array($fieldType, ['password'])) {
            if ($fieldType == 'checkbox') {
                if (!empty($field->getValue())) {
                    $attrs['value'] = self::clearString($field->getValue());
                }
            } else {
                $attrs['value'] = self::clearString($field->getValue());
            }
        }
        if ($fieldType == 'checkbox' && $field->getValue() == true) {
            $attrs['checked'] = 'checked';
        }
        if ($fieldType == 'number') {
            if (!method_exists($field, 'getStep')) {
                throw new InvalidArgumentException('Field ' . get_class($field) . ' does not have method getStep');
            }
            if (!method_exists($field, 'getMinValue')) {
                throw new InvalidArgumentException('Field ' . get_class($field) . ' does not have method getMinValue');
            }
            if (!method_exists($field, 'getMaxValue')) {
                throw new InvalidArgumentException('Field ' . get_class($field) . ' does not have method getMaxValue');
            }
            if ($field->getStep() > 0) {
                $attrs['step'] = str_replace(',', '.', $field->getStep());
                $min = $field->getMinValue();
                if (isset($min)) {
                    $attrs['min'] = str_replace(',', '.', $min);
                }
                $max = $field->getMaxValue();
                if (isset($max)) {
                    $attrs['max'] = str_replace(',', '.', $max);
                }
            }
        }
        $attrs = self::getAttrs($attrs, $field, $params['attributes'] ?? []);
        $properties = '';
        if (!isEmpty($field->getProperties())) {
            $properties = implode(' ', $field->getProperties());
        }
        $multiple = '';
        if ($field instanceof Fields\MultipleEmailField) {
            $multiple = ' multiple';
        }
        return '<input type="' . $fieldType . '" ' . implode(' ', $attrs) . $properties . $multiple . '/>' . "\n";
    }

    /**
     * ocisti/nahradi string o skarede znaky
     *
     * @param null|string $string
     */
    public static function clearString(?string $string): string
    {
        return htmlspecialchars($string ?? '');
    }

    /**
     *
     * @param array<string, int|float|string> $attrs atributy z definice formulare
     * @param BaseField $field
     * @param array<string, int|float|string> $params parametry/atributy, ktere byly predany fci getField
     * @return string[]
     */
    protected static function getAttrs(array $attrs, BaseField $field, array $params = [])
    {
        $str = [];
        $attrs = self::getFieldAttributeArray($attrs, $field, $params);
        foreach ($attrs as $key => $value) {
            $str[] = $key . '="' . $value . '"';
        }
        return $str;
    }

    protected static function getFieldAttributeArray(array $_attrs, BaseField $field, array $params = [])
    {
        $fieldAttributes = $field->getAttributes();
        if (array_key_exists('class', $fieldAttributes)) {
            // implode classes to string
            $fieldAttributes['class'] = implode(" ", $fieldAttributes['class']);
        }
        $attrs = self::mergeAttributes(self::mergeAttributes($_attrs, $fieldAttributes), $params);
        //$attrs = self::mergeAttributes($attrs, $params);
        if ($field->hasErrors()) {
            $attrs['class'] .= ' error';
        }
        if ($field->isRequired()) {
            $attrs['required'] = 'required';
        }
        $attrs['name'] = $field->getFormName();
        return $attrs;
    }

    protected static function mergeAttributes(array $patternAttrs, array $paramAttrs = [])
    {
        if (!empty($paramAttrs)) {
            $cssClasses = [
                $patternAttrs['class'] ?? '',
                $paramAttrs['class'] ?? ''
            ];
            $cssClasses = array_filter($cssClasses);
            $css = implode(' ', $cssClasses);
            $patternAttrs = array_merge($patternAttrs, $paramAttrs);
            $patternAttrs['class'] = $css;
        }
        return $patternAttrs;
    }

    /**
     *
     * @param array $pattern
     * @param SelectField $field
     * @param array<string, mixed> $params
     * @return string
     */
    protected static function parseSelect(array $pattern, SelectField $field, ?array $params = [])
    {
        $attrs = self::getAttrs($pattern['attributes'], $field, $params['attributes'] ?? []);
        $optionsHtml = "";
        if ($field->hasUnselectedText()) {
            $optionsHtml .= '<option value=""'
                . (!empty($field->getValue()) ? ' selected="selected"' : '')
                . '>'
                . $field->getUnselectedText()
                . '</option>';
        }

        foreach ($field->getOptionsStructure() as $optionStructure) {
            if ($optionStructure['type'] == SelectField::OPTGROUP_TYPE) {
                $optionsHtml .= '<optgroup label="' . $optionStructure["label"] . '">';
                foreach ($optionStructure["options"] as $optVal) {
                    $label = $optVal['label'];
                    if ($optVal['escape']) {
                        $label = htmlentities($optVal['label']);
                    }
                    $optionsHtml .= '<option value="' . $optVal['key'] . '"' .
                        ($optVal["selected"] ? ' selected="selected"' : '') . '>' .
                        $label
                        . '</option>';
                }
                $optionsHtml .= '</optgroup>';
            } elseif ($optionStructure['type'] == SelectField::OPTION_TYPE) {
                $label = $optionStructure['label'];
                if ($optionStructure['escape']) {
                    $label = htmlentities($optionStructure['label']);
                }
                $optionsHtml .= '<option value="' . $optionStructure['key'] . '"' . ($optionStructure["selected"] ? ' selected="selected"' : '') . '>'
                    . $label
                    . '</option>';
            }
        }
        if ($field->hasAttribute('readonly') || $field->hasProperty('readonly')) {
            $attrs['readonly'] = 'readonly';
        }
        if ($field->hasAttribute('disabled') || $field->hasProperty('disabled')) {
            $attrs['disabled'] = 'disabled';
        }

        return '<select ' . implode(' ', $attrs) . '>' . $optionsHtml . '</select>' . "\n";
    }

    /**
     *
     * @param array<string, mixed> $pattern
     * @param BaseField $field
     * @param array<string, mixed> $params
     * @return string
     */
    protected static function parseRadio(array $pattern, BaseField $field, array $params = [])
    {
        if (!method_exists($field, 'getOptions')) {
            throw new InvalidArgumentException('Field ' . get_class($field) . ' does not have method getOptions');
        }

        if (isset($params['attributes']['id'])) {
            unset($params['attributes']['id']);
        }
        $attrs = self::getAttrs($pattern['attributes'] ?? [], $field, $params['attributes'] ?? []);
        $labelPos = $pattern['textPosition'];
        if (empty($labelPos)) {
            $labelPos = self::LABEL_TEXT_AFTER_FIELD;
        }
        $separator = !empty($params['separator']) ? $params['separator'] : $pattern['separator'];
        if (empty($separator)) {
            $separator = ' ';
        }
        $captionClass = null;
        if (
            array_key_exists('caption', $pattern)
            && is_array($pattern['caption'])
            && array_key_exists('class', $pattern['caption'])
        ) {
            $captionClass = $pattern['caption']['class'];
        }
        if (array_key_exists('caption', $params)
            && is_array($params['caption'])
            && array_key_exists('class', $params['caption'])
        ) {
            $captionClass = $params['caption']['class'];
        }
        $options = [];
        foreach ($field->getOptions() as $key => $value) {
            $swap = '<input ' .
                ((string)$key == (string)$field->getValue() ? ' checked="checked" ' : '') .
                'value="' . $key . '" ' .
                'type="radio" ' . implode(' ', $attrs) . '>';
            $caption = '<span' . (!empty($captionClass) ? ' class="' . $captionClass . '"' : '') . '>' . $value . '</span>';
            if ($labelPos == self::LABEL_TEXT_AFTER_FIELD) {
                $swap .= $caption;
            } else {
                $swap = $caption . $swap;
            }
            $labelClass = 'form__item--radio';
            if (isset($pattern['label']['class'])) {
                $labelClass = $pattern['label']['class'];
            }
            if (isset($params['label']['class'])) {
                $labelClass .= ' ' . $params['label']['class'];
            }
            $options[] = '<label class="' . $labelClass . '">' . $swap . ' </label>';
        }
        return implode($separator, $options);
    }

    /**
     *
     * @param array $pattern
     * @param CheckboxListField $field
     * @param array $params
     * @return string
     */
    protected static function parseCheckboxListField(array $pattern, CheckboxListField $field, array $params = []): string
    {
        $attrs = self::getAttrs($pattern['attributes'] ?? [], $field, $params['attributes'] ?? []);
        $labelPos = isset($pattern['textPosition']) ? $pattern['textPosition'] : null;
        if (empty($labelPos)) {
            $labelPos = self::LABEL_TEXT_AFTER_FIELD;
        }
        if (isset($params['separator']) && !empty($params['separator'])) {
            $separator = $params['separator'];
        } elseif (isset($pattern['separator'])) {
            $separator = $pattern['separator'];
        }
        if (empty($separator)) {
            $separator = ' ';
        }

        $ret = [];
        foreach ($field->getOptions() as $key => $value) {
            $captionAttr = [];
            if (isset($pattern['caption']) || isset($params['caption'])) {
                $captionParams = array_merge($pattern['caption'] ?? [], $params['caption'] ?? []);
                foreach ($captionParams as $attr => $attrValue) {
                    $captionAttr[] = sprintf('%s="%s"', $attr, strip_tags($attrValue));
                }
            }

            $swap = '<input ' .
                ($value["checked"] ? 'checked="checked"' : '') .
                'name="' . $field->getFormName() . "[" . $key . ']" ' .
                'value="' . $key . '" ' .
                'type="checkbox" >';
            $caption = '<span ' . implode(' ', $captionAttr) . '>' . $value['label'] . '</span>';
            if ($labelPos == self::LABEL_TEXT_AFTER_FIELD) {
                $swap .= $caption;
            } else {
                $swap = $caption . $swap;
            }

            $labelClass = '';
            if (isset($pattern['label']['class'])) {
                $labelClass .= $pattern['label']['class'];
            }
            if (isset($params['label']['class'])) {
                $labelClass .= $params['label']['class'];
            }
            $ret[] = '<label class="' . $labelClass . '">' . $swap . ' </label>';
        }

        return implode($separator, $ret);
    }

    /**
     *
     * @param array $pattern
     * @param BaseField $field
     * @param array $params
     * @return string
     */
    protected static function parseTextArea($pattern, Fields\BaseField $field, $params = [])
    {
        $attrs = self::getAttrs($pattern['attributes'] ?? [], $field, $params['attributes'] ?? []);

        if ($field->hasAttribute('readonly')) {
            $attrs['readonly'] = 'readonly';
        }
        if ($field->hasAttribute('disabled')) {
            $attrs['disabled'] = 'disabled';
        }
        $properties = '';
        if (!isEmpty($field->getProperties())) {
            $properties = implode(' ', $field->getProperties());
        }

        return '<textarea ' . implode(' ', $attrs) . $properties . '>' . clearString($field->getValue()) . '</textarea>' . "\n";
    }

    protected static function parseHiddenField(Fields\BaseField $field, $params = [])
    {
        $attrs = [];
        foreach ($field->getAttributes() as $key => $value) {
            $attrs[] = $key . '="' . $value . '"';
        }
        if (array_key_exists('attributes', $params)) {
            foreach ($params["attributes"] as $key => $value) {
                $attrs[] = $key . '="' . $value . '"';
            }
        }
        return '<input type="hidden" value="'
            . $field->getValue()
            . '" name="'
            . $field->getFormName()
            . '"'
            . (!empty($attrs) ? ' ' . implode(' ', $attrs) : '')
            . '>';
    }

    /**
     *
     * @param array $settings
     * @param array $params
     * @return string
     */
    protected static function generateWrapper(array $settings, array $params = [])
    {
        $settings = self::applyPatternChanges(
            $settings,
            $params,
            [
                'tag' => 'label',
                'captionPosition' => self::LABEL_TEXT_BEFORE_FIELD
            ]
        );
        $attrs = self::mergeAttributes($settings['attributes'] ?? [], $params['attributes'] ?? []);
        $str = [];
        if (!empty($attrs) && is_array($attrs)) {
            foreach ($attrs as $key => $value) {
                $str[] = $key . '="' . $value . '"';
            }
        }
        $ret = '<' . $settings['tag'] . ' ' . implode(' ', $str) . '>__CONTENT__</' . $settings['tag'] . '>';
        return $ret;
    }

    /**
     * Aplikuje defaultni hodnoty a zmeny predane pres parametr vykreslovaci fce oproti vzoru
     *
     * @param array $settings
     * @param array $params
     * @param array $defaults
     * @return array
     */
    protected static function applyPatternChanges(array $settings, array $params, array $defaults = [])
    {
        $settings = self::setDefaults($settings, $defaults);
        foreach ($params as $key => $value) {
            if ($key == 'attributes') {
                continue;
            }
            if (array_key_exists($key, (array)$settings) && $settings[$key] != $params[$key]) {
                $settings[$key] = $params[$key];
            }
        }
        return $settings;
    }

    /**
     * sets defaults to $settings from $defaults if key does not exist in $settings
     * @param array $settings
     * @param array $defaults
     * @return array
     */
    protected static function setDefaults(array $settings, array $defaults)
    {
        foreach ($defaults as $key => $value) {
            if (!array_key_exists($key, $settings)) {
                $settings[$key] = $value;
            }
        }
        return $settings;
    }

    /**
     *
     * @param array<string, mixed> $settings
     * @param string $text
     * @param array<string, mixed> $params
     * @return string
     */
    protected static function generateCaption(array $settings, string $text, array $params = [])
    {
        $settings = self::applyPatternChanges($settings, $params, ['tag' => 'span']);

        // add class to attributes
        $settings['attributes'] = self::setDefaults(
            $settings['attributes'] ?? [],
            ['class' => 'form__item__label']
        );

        $attrs = self::mergeAttributes(
            $settings['attributes'],
            $params['attributes'] ?? []
        );
        $str = [];
        if (!empty($attrs) && is_array($attrs)) {
            foreach ($attrs as $key => $value) {
                $str[] = $key . '="' . $value . '"';
            }
        }

        return '<' . $settings['tag'] . ' ' . implode(' ', $str) . '>' . $text . '</' . $settings['tag'] . '>';
    }

    protected static function generateFieldError(array $settings, array $errors = [], array $params = [])
    {
        $settings = self::applyPatternChanges($settings, $params, ['tag' => 'div', 'separator' => '<br>']);
        $settings['attributes'] = self::setDefaults($settings['attributes'], ['class' => 'message']);

        $attrs = self::mergeAttributes($settings['attributes'], $params['attributes'] ?? []);

        $str = [];
        if (!empty($attrs) && is_array($attrs)) {
            foreach ($attrs as $key => $value) {
                $str[] = $key . '="' . $value . '"';
            }
        }
        return '<'
            . $settings['tag']
            . ' '
            . implode(' ', $str)
            . '>'
            . implode($settings['separator'], $errors)
            . '</'
            . $settings['tag']
            . '>';
    }
}
