<?php

namespace IZON\Forms\Fields;

use InvalidArgumentException;
use IZON\Forms\ValidationRule;

/**
 * Pole odpovidajici rozsahu cisel ve kterem se da posunovat
 * https://www.w3.org/wiki/HTML/Elements/input/range
 */
class IntegerRangeField extends BaseField
{
    /**
     * @var int minimalni hodnota, ktera muze byt do pole zadana
     */
    protected int $minValue;

    /**
     * @var int maximalni hodnota, ktera muze byt do pole zadana
     */
    protected int $maxValue;

    /**
     * @var int
     */
    protected int $step = 1;

    /**
     * @var null|array pole s moznymi hodnotami soupatoru
     */
    protected ?array $valuesList = null;

    /**
     *
     * @param string $name
     * @param int $minValue
     * @param int $maxValue
     * @param string $label
     */
    public function __construct(
        string $name,
        int $minValue,
        int $maxValue,
        string $label = ''
    ) {
        $this->name = $name;
        $this->propertyName = $this->name;
        $this->label = $label;

        if ($maxValue < $minValue) {
            throw new InvalidArgumentException("minValue = " . $minValue . " nesmi byt vetsi nez maxValue = " . $maxValue);
        }

        $this->minValue = $minValue;
        $this->maxValue = $maxValue;

        $rule = new ValidationRule();
        $rule->setMessage(_('Zadaná hodnota je mimo rozsah.'));

        $function = function (IntegerRangeField $field, $args = []) {
            return $field->getMinValue() <= $field->getValue() // je mensi nez minimum
                && $field->getValue() <= $field->getMaxValue(); // je vetsi nez maximum
        };
        $rule->setValidator($function);
        $this->addRule($rule);
    }

    public function getMinValue(): int
    {
        return $this->minValue;
    }

    public function setMinValue(int $minValue): IntegerRangeField
    {
        $this->minValue = $minValue;
        return $this;
    }

    public function getMaxValue(): int
    {
        return $this->maxValue;
    }

    public function setMaxValue(int $maxPrice): IntegerRangeField
    {
        $this->maxValue = $maxPrice;
        return $this;
    }

    public function parse()
    {
        return $this->value;
    }

    public function serialize($value)
    {
        $this->value = $value;
    }

    public function getStep(): int
    {
        return $this->step;
    }

    public function setStep(int $step): IntegerRangeField
    {
        if ($step <= 0) {
            throw new InvalidArgumentException("Step must be greater than 0");
        }
        $this->step = $step;
        return $this;
    }

    protected function getSpecificSupportedProperties(): array
    {
        return [];
    }
}
