<?php

namespace IZON\Forms\Fields;

use InvalidArgumentException;

/**
 * Pole odpovidajici input text
 */
class FloatField extends BaseField
{
    protected float $step = 0.00000001;

    /**
     * @var float minimalni hodnota, ktera muze byt do pole zadana
     */
    protected ?float $minValue = null;

    /**
     * @var float maximalni hodnota, ktera muze byt do pole zadana
     */
    protected ?float $maxValue = null;

    /**
     *
     * @param string $name
     * @param string $label
     */
    public function __construct(string $name, string $label = '')
    {
        $this->name = $name;
        $this->label = $label;
        $this->propertyName = $this->name;
    }

    public function parse()
    {
        if ($this->nullOnEmpty && empty(trim($this->value))) {
            return null;
        }
        return floatval($this->value);
    }

    public function serialize($value)
    {
        if (is_null($value) || (is_string($value) && $value == '')) {
            $this->value = '';
            return;
        }

        // shift value to nearest step
        $reminder = fmod($value, $this->step);
        if ($reminder > $this->step / 2) {
            $reminder -= $this->step;
        }
        $value -= $reminder;

        // display only decimals necessary for step
        $decimals = -1 * intval(log10($this->step));
        $this->value = number_format($value, $decimals, '.', '');
    }

    public function getStep(): float
    {
        return $this->step;
    }

    public function setStep(float $step): FloatField
    {
        if ($step <= 0) {
            throw new InvalidArgumentException("Step must be greated than 0");
        }
        $this->step = $step;
        return $this;
    }

    public function getMinValue(): ?float
    {
        return $this->minValue;
    }

    /**
     * minimalni hodota, ktera muze byt do pole zadana
     * @param null|float $minValue null to set no minimum
     * @return FloatField
     */
    public function setMinValue(?float $minValue): FloatField
    {
        $this->minValue = $minValue;
        return $this;
    }

    public function getMaxValue(): ?float
    {
        return $this->maxValue;
    }

    /**
     * maximalni hodota, ktera muze byt do pole zadana
     * @param null|float $maxValue null to set no maximum
     * @return FloatField
     */
    public function setMaxValue(?float $maxValue): FloatField
    {
        $this->maxValue = $maxValue;
        return $this;
    }

    protected function getSpecificSupportedProperties(): array
    {
        return [];
    }
}
