<?php
namespace IZON\Forms;

use Exception;
use IZON\DB\EntityManagerInterface;
use IZON\Forms\Fields\EntityHiddenField;
use ReflectionClass;

use IZON\Forms\Fields\TransientField;
use IZON\Logs\Logger;

use IZON\Forms\Fields\BaseField;
use IZON\Forms\Fields\CharField;
use IZON\Forms\Fields\CheckboxField;
use IZON\Forms\Fields\CheckboxListField;
use IZON\Forms\Fields\DateFiled;
use IZON\Forms\Fields\DateTimeFiled;
use IZON\Forms\Fields\EmailField;
use IZON\Forms\Fields\FileField;
use IZON\Forms\Fields\FloatField;
use IZON\Forms\Fields\FloatRangeField;
use IZON\Forms\Fields\HiddenField;
use IZON\Forms\Fields\IdField;
use IZON\Forms\Fields\ImageField;
use IZON\Forms\Fields\IntegerField;
use IZON\Forms\Fields\IntegerRangeField;
use IZON\Forms\Fields\PasswordField;
use IZON\Forms\Fields\PhoneField;
use IZON\Forms\Fields\RadioListField;
use IZON\Forms\Fields\SelectField;
use IZON\Forms\Fields\TextField;


/**
 * Trida formulare
 */
class Form implements FormInterface {

    /**
     * pri edit/flush hodont do/z formulare pristupovat primo na property
     */
    const EDIT_FLUSH_ACCESS_PROPERTIES = 1;

    /**
     * pri edit/flush hodont do/z formulare pristupovat pres gettery a settery
     */
    const EDIT_FLUSH_ACCESS_GETTERS_SETTERS = 2;

    /**
     * jednotliva pole formulare
     * asociativni pole, klice jsou name poli, klice jsou jednotliva pole
     * @var array BaseField fole formulare
     */
    protected $fileds;

    /**
     * chyby primo vazane na formular
     * @var array
     */
    protected $errors;

    /**
     * validator pro cely formular
     * muze provadet validace zavisle na vice polich
     * @var callable
     */
    protected $validator;

    /**
     * prefix slouzi k zanoreni do pole predavaneho v set values a ke spravnemu vypisu
     * formName pro jednotliva pole ve formulari
     * @var array prefix pro vsechny pole formulare
     */
    protected $prefix;

    /**
     * @var Logger
     */
    protected $log;


    public function __construct(array $prefix = []) {
        if( $prefix == NULL ) { // TODO: NEWVERSION je kvuli spetne kompatibilite, pri vytvoreni nove verze odebrat a vyzadovat array
            $prefix = [];
        }
        if( !is_array($prefix) ) {
            throw  new Exception("prefix musi byt pole");
        }
        $this->prefix = $prefix;

        $this->fileds = [];
        $this->errors = [];

        $this->log = Logger::getLogger(__CLASS__);
    }

    /**
     * @return array reprezentuje preffix
     */
    public function getPrefix() {
        return $this->prefix;
    }

    public function setPrefix(array $prefix) {
        $this->prefix = $prefix;
        foreach($this->fileds as $field) {
            $field->setPrefix($prefix);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function setValues(array $array) {
        $prefixedArray = $array;
        foreach($this->prefix as $pref) {
            $prefixedArray = $prefixedArray[$pref];
        }
        foreach($this->fileds as $name => $field) {
            if( isset($prefixedArray[$name]) ) {
                $field->setValue($prefixedArray[$name]);
            } else {
                $field->setValue(NULL);
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getValues(): array {
        $return = [];
        foreach ($this->fileds as $name => $field) {
            $return[$name] = $field->getValue();
        }
        // add prefix
        foreach(array_reverse($this->prefix) as $prefixPart) {
            $return = [$prefixPart => $return];
        }
        return $return;
    }

    /**
     * provede validaci formulare jako celku
     * @return boolean
     */
    public function validate() {
        // validuje jednotliva pole
        foreach ($this->fileds as $name => $field) {
            $field->validate();
        }
        // vola validator celeho formulare
        if( $this->validator ) {
            $validator = $this->validator;
            $validator($this);
        }
    }

    /**
     * zjisti jestli ma formular nebo nejake formularove pole chybu
     * @return boolean
     */
    public function hasErrors() {
        $hasErrors = false;
        /* @var $field BaseField */
        foreach ($this->fileds as $name => $field) {
            $hasErrors = $hasErrors || $field->hasErrors();
        }
        return $hasErrors || !empty($this->errors);
    }

    public function hasFormErrors() {
      return !empty($this->errors);
    }

    /**
     * prida chybu navazanou primo na formulari ne navazane na formularove pole
     * @param string $errorMessage
     */
    function addFormError($errorMessage) {
        $this->errors[] = $errorMessage;
    }

    /**
     * vrati chyby navazane primo na formulari ne navazane na formularove pole
     * @return array
     */
    function getFormErrors() {
        return $this->errors;
    }

    /**
     * vrati formularove pole
     * @param string $name
     * @return BaseField
     */
    public function getField($name) {
        if (!isset($this->fileds[$name])) {
            throw new \Exception("Pole $name není definováno.");
        } else {
            return $this->fileds[$name];
        }
    }

    /**
     * Varci pole zaregistrovanych formularovych poli
     *
     * @return BaseField
     */
    function getFileds() {
        return $this->fileds;
    }

    /**
     * Chceck if field exist in form
     * @param string $name name of field
     * @return boolean True - field exist False - field is not exist
     */
    function haveField($name){
        if (isset($this->fileds[$name])) {
            return True;
        } else {
            return False;
        }
    }
    /**
     *
     * prida do formulare jakekoli pole
     * @param BaseField $filed
     * @return BaseField
     * @throws Exception
     */
    function addFiled(BaseField $filed) {
        if( !array_key_exists($filed->getName(), $this->fileds) ) {
            $this->fileds[$filed->getName()] = $filed;
            $filed->setPrefix($this->prefix);
            return $this->fileds[$filed->getName()];
        } else {
            throw new \Exception('FIELD EXISTS', 100);
        }
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return IdField
     */
    public function addIdField($fieldName, $fieldLabel = '') {
        $field = new IdField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakywebhom jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return HiddenField
     */
    public function addHiddenField($fieldName, $fieldLabel = '') {
        $field = new HiddenField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return CharField
     */
    public function addCharField($fieldName, $fieldLabel = '') {
        $field = new CharField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Prida do formulare dlouhe textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return TextField
     */
    public function addTextField($fieldName, $fieldLabel = '') {
        $field = new TextField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return CharField
     */
    public function addCheckboxField($fieldName, $fieldLabel = '') {
        $field = new CheckboxField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return PasswordField
     */
    public function addPasswordField($fieldName, $fieldLabel = '') {
        $field = new PasswordField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return DateFiled Description
     */
    public function addDateFiled($fieldName, $fieldLabel = '') {
        $field = new DateFiled($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return DateTimeFiled Description
     */
    public function addDateTimeFiled($fieldName, $fieldLabel = '') {
        $field = new DateTimeFiled($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Adds file
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return FileField
     */
    public function addFileField($fieldName, $fieldLabel = '') {
        $field = new FileField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Adds file
     * @param string $fieldName
     * @param string $fieldLabel
     * @return FileField
     */
    public function addImageField($fieldName, $fieldLabel = '') {
        $field = new ImageField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Adds intefer field
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return FloatField
     */
    public function addFloatField($fieldName, $fieldLabel = '') {
        $field = new FloatField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Adds imteger field
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return IntegerField
     */
    public function addIntegerField($fieldName, $fieldLabel = '') {
        $field = new IntegerField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * Adds file
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return EmailField
     */
    public function addEmailField($fieldName, $fieldLabel = '') {
        $field = new EmailField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }
    /**
     * Adds multiple email field
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return Fields\MultipleEmailField
     */
    public function addMultipleEmailField($fieldName, $fieldLabel = '') {
        $field = new Fields\MultipleEmailField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }
    /**
     * Adds field
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel label pro prvek
     * @return PhoneField
     */
    public function addPhoneField($fieldName, $fieldLabel = '') {
        $field = new PhoneField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }
    /**
     *  pridava select pole
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return SelectField
     */
    public function addSelectField($fieldName, $fieldLabel = '') {
        $field = new SelectField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     *  pridava radio pole
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return RadioListField
     */
    public function addRadioListField($fieldName, $fieldLabel = '') {
        $field = new RadioListField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     *  adds array of checkoxes with multiple possible selected options
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return CheckboxListField
     */
    public function addCheckboxListField($fieldName, $fieldLabel = '') {
        $field = new CheckboxListField($fieldName, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * pole integeru, ktere ma rozsah odnot od nejmensi do nejvetsi
     * @param string $fieldName
     * @param integer $minValue
     * @param integer $maxValue
     * @param string $fieldLabel
     * @return IntegerRangeField
     */
    public function addIntegerRangeField($fieldName, $minValue, $maxValue, $fieldLabel = '') {
        $field = new IntegerRangeField($fieldName, $minValue, $maxValue, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    /**
     * pole floatu, ktere ma rozsah odnot od nejmensi do nejvetsi
     * @param string $fieldName
     * @param integer $minValue
     * @param integer $maxValue
     * @param string $fieldLabel
     * @return FloatRangeField
     */
    public function addFloatRangeField($fieldName, $minValue, $maxValue, $fieldLabel = '') {
        $field = new FloatRangeField($fieldName, $minValue, $maxValue, $fieldLabel);
        $this->addFiled($field);
        return $field;
    }

    public function addEntityHiddenField(string $fieldName, EntityManagerInterface $entityManager, string $entityClass) : EntityHiddenField {
        $field = new EntityHiddenField($fieldName, $entityManager, $entityClass);
        $this->addFiled($field);
        return $field;
    }

    /**
     * ze zadaneho objektu nastavi hodnoty jednotlivych formularovych poli
     * @param mixed $object z jakeho objektu se maji ziskat data
     * @param integer $accessType Jakym zpusobem se ma pristupovat k property v objektu. Jestli primo nebo pres gettery a settery, default je dvuli kompatibilite pres gettery a settery
     */
    public function edit($object, $accessType = self::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) {
        if( is_null($object) ) {
            throw new Exception("Objekt ze ktereho se data kopiruji nesmi byt null");
        }

        /* @var $reflectionClass ReflectionClass */
        $reflectionClass = new ReflectionClass($object);
        /* @var $field BaseField */
        foreach ($this->fileds as $key => $field) {
            if( $accessType == self::EDIT_FLUSH_ACCESS_PROPERTIES ) { // pristupovat pres property
                if( $reflectionClass->hasProperty($field->getPropertyName()) ) {
                    $reflectionProperty = $reflectionClass->getProperty($field->getPropertyName());
                    $reflectionProperty->setAccessible(true);
                    $field->serialize($reflectionProperty->getValue($object));
                } else if( method_exists($object, "__isset")
                            && $object->__isset($field->getPropertyName()) ) { // ma property nadefinovane pomoci __isset, __get a __set metod
                    $propertyName = $field->getPropertyName();
                    $field->serialize( $object->$propertyName );
                } else {
                    $this->log->info("Doesn't have property ". $field->getPropertyName() ." to get data from");
                }
            } else if( $accessType == self::EDIT_FLUSH_ACCESS_GETTERS_SETTERS ) { // pristupovat pres gettery a settery
                $propertyName = $field->getPropertyName();
                $getterName = \IZON\Object\createGetterMethodName($propertyName);
                $isName = \IZON\Object\createIsMethodName($propertyName);

                if( $reflectionClass->hasMethod($getterName) ) { // has getter method get data through it
                    $getterMethod = $reflectionClass->getMethod($getterName);
                    $value = $getterMethod->invoke($object);
                    $field->serialize($value);
                } else if( $reflectionClass->hasMethod($isName) ) {
                    $getterMethod = $reflectionClass->getMethod($isName);
                    $value = $getterMethod->invoke($object);
                    $field->serialize($value);
                } if( $reflectionClass->hasMethod("__call" ) // has magic method __call to be able to call getter method implemented by __call method
                            && $reflectionClass->hasMethod("__isset" ) // has method __isset to determine if setter for property should be provided by __call method
                            && $object->__isset($propertyName) ) { // has property so __call should provide getter

                    $value = $object->__call($getterName, []);
                    $field->serialize($value);
                } else {
                    $this->log->info("Doesn't have getter ". $getterName ." to get data from");
                }
            } else {
                throw new Exception("Unsupported access type ". $accessType);
            }
        }
    }

    /**
     * z hodnot formularovych poli nastavi hodnoty poli v objektu
     */

    /**
     *
     * @param object $object
     * @param integer $accessType Jakym zpusobem se ma pristupovat k property v objektu. Jestli primo nebo pres gettery a settery, default je dvuli kompatibilite pres gettery a settery
     * @throws Exception
     */
    public function flush($object, $accessType = self::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) {
        if( is_null($object) ) {
            throw new Exception("Objekt do ktereho se data kopiruji nesmi byt null");
        }

        /* @var $reflectionClass ReflectionClass */
        $reflectionClass = new ReflectionClass($object);
        /* @var $field BaseField */
        foreach ($this->fileds as $key => $field) {
            if( $field->isOneWay() ) { // z oneway pole se nemapuji informace zpet do objektu
                continue;
            }

            if( $accessType == self::EDIT_FLUSH_ACCESS_PROPERTIES ) { // pristupovat pres property
                if ($reflectionClass->hasProperty($field->getPropertyName())) {
                    $reflectionProperty = $reflectionClass->getProperty($field->getPropertyName());
                    $reflectionProperty->setAccessible(true);
                    $reflectionProperty->setValue($object, $field->parse());
                } else if( $reflectionClass->hasMethod("__isset") // ma metodu isset
                            && $object->__isset($field->getPropertyName()) ) { // ma property nadefinovane pomoci __isset, __get a __set metod
                    $propertyName = $field->getPropertyName();
                    $object->$propertyName = $field->parse();
                } else {
                    $this->log->info("Doesn't have property ". $field->getPropertyName() ." to flush data to");
                }
            } else if( $accessType == self::EDIT_FLUSH_ACCESS_GETTERS_SETTERS ) { // pristupovat pres gettery a settery
                $propertyName = $field->getPropertyName();
                $setterName = \IZON\Object\createSetterMethodName($propertyName);

                if( $reflectionClass->hasMethod($setterName) ) { // ma setter metodu nastavit hodnotu pres ni
                    $setterMethod = $reflectionClass->getMethod($setterName);
                    $setterMethod->invoke($object, $field->parse());
                } else if($reflectionClass->hasMethod("__call" ) // has magic method __call to be able to call getter method implemented by __call method
                            && $reflectionClass->hasMethod("__isset" ) // has method __isset to determine if setter for property should be provided by __call method
                            && $object->__isset($field->getPropertyName()) ) { // has property so __call should provide getter
                    $object->__call($setterName, [$field->parse()]);
                } else {
                    $this->log->info("Doesn't have setter ". $setterName ." to flush data to");
                }
            } else {
                throw new Exception("Unsupported access type ". $accessType);
            }
        }
    }

    /**
     * sets validator for whole form
     * @param callable $validator is a method with one parame Form $form, it sets found errors tirectly to $form
     */
    function setValidator($validator) {
        if( !is_callable($validator) ) {
            throw new Exception("Validator must be callable");
        }
        $this->validator = $validator;
    }

    /**
     * @return array
     */
    function getErrorsArray() {
        $errors = [];
        foreach ($this->getFileds() as $fild) {
            if ($fild->hasErrors()) {
                $errorsArray = $fild->getErrors();
                $errors[] = ["formName" => $fild->getFormName(), "errors" => $errorsArray];
            }
        }
        return $errors;
    }

    /**
     * Smazne pole z formulare
     *
     * @param string $key - index, pod kterym bylo pole pridano
     */
    public function removeField($key) {
        unset($this->fileds[$key]);
    }

    public function __clone() {
        // naklonuje formularova pole
        $clonedFields = [];
        foreach($this->fileds as $index => $field) {
            $clonedField = clone $field;
            $clonedFields[$index] = $clonedField;
        }
        $this->fileds = $clonedFields;
    }
}
