<?php

namespace IZON\Forms\Fields;

use \IZON\Forms;
use \IZON\Forms\ValidationRule;

/**
 * Zaklad fomularoveho pole
 */
abstract class BaseField {
    
    protected $errors;
    
    /**
     * true, pokud musi byt pritomne ve formulari
     * @var boolean 
     */
    protected $required;
    
    /**
     * Ma se pouzit label?<br>
     * Defaultne true
     * 
     * @var boolean
     * 
     * @deprecated jestli pouzit label se nastavuji ve view
     */
    protected $useLabel = true;
    
    /**
     * popiska formularoveho pole
     * @var string 
     */
    protected $label;
    
    /**
     * stylova trida popisky formularoveho pole
     * @var string 
     * 
     * @deprecated css tridy a se nastavuji ve view
     */
    protected $labelClass;
    
    /**
     * relativni nazev pole v danem formulari
     * podle ktereho se bude ziskavat hornota z request
     * @var string 
     */
    protected $name;
    
    /**
     * jaky prefix se ma pripojit pred name ve formulari
     * @var string 
     */
    protected $prefix = [];

    /**
     * nazev do ktere se ma hodnota furmularoveho prvku zkopirovat
     * defaultne se nastavuje jako name
     * 
     * v pripade ze je zanoreno do podobjektu tak se predava jako pole se jmeny zleva doprava
     * @var array 
     */
    protected $propertyName = NULL;

    /**
     * Pole s atributy tagu input, napr.: placeholder, readonly, selected atp.
     * 
     * @var array
     */
    protected $attributes = [];

    /**
     * hodnota ziskana z formulare pomoci POST, GET nebo RESULT pole
     * a takova, ktera se ma do html formulare vlozit
     * @var mixed 
     */
    protected $value;

    /**
     * defaultni hodnota pole
     * @var mixed 
     */
    protected $default;
    
    /**
     * validacni pravidla
     * @var \IZON\Forms\ValidationRule 
     */
    protected $rules = [];
    
    /**
     * @var boolean pokud je pole prazdne vloz null
     */
    protected $nullOnEmpty = false;
    
    /**
     * @var boolean rika jestli se ma hodnota z formulare zpet nastavovat z formulare do objektu
     */
    protected $readonly = false;


    /**
     * inicializuje pole hodnotou zislanou z POST, GET, REQUEST
     * @param mixed $value
     */
    public function setValue($value) {
        $this->value = $value;
    }
    
    /**
     * vraci hodnotu primo odpovidajici parametru ziskaneho z http
     */
    public function getValue() {
        return $this->value;
    }
    
    /**
     * provede validaci prvku
     */
    function validate() {
      if( $this->required 
          && $this->value == NULL) {
          $this->errors[] = _("Pole je povinné");
      }
      foreach($this->rules as $rule) {
        if(!$rule->validate($this)) {
          $this->errors[] = sprintf($rule->getMessage(), $rule->getArgs()) ;
        }
      }
      return $this->errors;
    }
    
    function addError($error) {
        $this->errors[] = $error;
    }
    
    function getErrors() {
      return $this->errors;
    }
    
    /**
     * vraci true pokud pole obsahuje chyby
     * @return booelan
     */
    function hasErrors() {
        return !empty($this->errors);
    }

    /**
     * parsuje vstup z formulaze do vnitrni podoby pouzivane v php
     * napriklad string date do objektu date
     * vraci tuto vytvorenou podobu
     */
    abstract function parse();
    
    /**
     * provede serializaci hodnoty do tvaru pouzitelneho v html formulari
     */
    abstract function serialize($value);
    
            
    function getRequired() {
      return $this->required;
    }

    function getLabel() {
      return $this->label;
    }
    public function getLabelClass() {
      return $this->labelClass;
    }

    
    function getName() {
      return $this->name;
    }
    
    /**
     * jak se ma jenovat pri vypisovani formulare, aby se spravne
     * navazalo pole _REQUEST 
     * @return string
     */
    function getFormName() {
        if( $this->prefix 
            && !empty($this->prefix) ) {
            $ret = "";
            foreach( $this->prefix as $key => $value) {
                if( $key == 0 ) {
                    $ret .= $value;
                } else {
                    $ret .= "[". $value ."]";
                }
            }
            return $ret ."[". $this->name ."]";
        }
        return $this->name;
    }
    
    /**
     * nastavi prefix
     * @param string $prefix
     */
    function setPrefix(array $prefix) {
        $this->prefix = $prefix;
        return $this;
    }
    
    function getPropertyName() {
      return $this->propertyName;
    }

    function getRules() {
      return $this->rules;
    }

    function setRequired($required = true) {
      $this->required = $required;
      return $this;
    }
    function isRequired() {
      return $this->required;
    }

    /**
     * @param string $label
     * @param string $class - bude odstraneno, lable class se nastavuje ve view
     * @return $this
     */
    function setLabel($label, $class = null) {
      $this->label = $label;
      $this->labelClass = $class;
      return $this;
    }
    
    /**
     * @deprecated css tridy a se nastavuji ve view
     * @param type $labelClass
     * @return BaseField
     */
    public function setLabelClass($labelClass) {
      $this->labelClass = $labelClass;
      return $this;
    }

    
    function setName($name) {
      $this->name = $name;
      return $this;
    }

    function setPropertyName($propertyName) {
      $this->propertyName = $propertyName;
      return $this;
    }

    function addRule(ValidationRule $rule) {
      $rule->setField($this);
      $this->rules[] = $rule;
      return $this;
    }
    
    function getDefault() {
      return $this->default;
    }

    function setDefault($default) {
      $this->default = $default;
    }

    function getAttributes() {
      return $this->attributes;
    }
    
    public function hasAttribute($name) {
        return isset($this->attributes[$name]);
    }

    public function getAttribute($name) {
      return $this->attributes[$name];
    }

    /**
     * Vrati atributy pole pro vypis do html
     * 
     * @return string
     */
    public function getAttributesAsString() {
        $ret = [];
        foreach($this->attributes as $key => $value) {
            if ($key == 'class') { // property class obsahuje pole trid
                $value = implode(" ", $value);
            }
            $ret[] = $key . '="' . $value . '"';
        }
        if ($this->required) {
            $ret[] = 'required="required"';
        }
        return implode(' ', $ret);
    }

    /**
     * Nastavuje jednotlive atributy pro input.<br>
     * Pro class a id jsou udelane samostane metody.
     * 
     * @param string $key
     * @param string|number $value
     * @return BaseField
     */
    public function addAttribute($key, $value) {
      $reject = ['class', 'id'];
      if(in_array($key, $reject)) {
        throw new \Exception('For atributes '. strtoupper(implode(', ', $reject)) .' use their own setters.');
      }
      $this->attributes[$key] = $value;
      return $this;
    }
    
    /**
     * Prida stylovou tridu
     * @deprecated css tridy a se nastavuji ve view
     * @param string $param - je string s nazvem tridy, ktera se ma pridat
     * @return BaseField
     */
    public function addClass($param) {
        if( !is_string($param) ) {
            throw new Exception("Parametr musi byt string s nazvem");
        }
        if( array_key_exists('class', $this->attributes) ) {
            // TODO: tuhle vetev vyresit nejak lepe
            $param = [$param];
            if( !empty($param) && is_array($param) ) {
                $this->attributes['class'] = array_unique(array_merge($this->attributes['class'], $param));
            }
        } else {
            $this->attributes['class'] = [$param];
        }
        return $this;
    }

    /**
     * Prida stylove tridy
     * @deprecated css tridy a se nastavuji ve view
     * @param mixed $param - bud pole stylovych trid, nebo string trid oddelenych mezerou, ie: 'trida01 trida02 trida03'
     * @return BaseField
     */
    public function addClasses($param) {
        if(!is_array($param)) {
          $param = explode(' ', trim($param));
        }
        if(array_key_exists('class', $this->attributes)) {
          if(!empty($param) && is_array($param)) {
            $this->attributes['class'] = array_unique( array_merge( $this->attributes['class'], $param) );
          }
        } else {
           $this->attributes['class'] = $param;
        }
      return $this;
    }
    
    /**
     * Nastavi id prvku
     * @deprecated css tridy a se nastavuji ve view
     * @param string $param
     * @return BaseField
     */
    public function setId($param) {
      $this->attributes['id'] = $param;
      return $this;
      
    }
    /**
     * Vrati id prvku
     * @deprecated css tridy a se nastavuji ve view
     * @return mixed [string|null]
     */
    public function getId() {
      if(isset($this->attributes['id'])) {
        return $this->attributes['id'];
      }
      return null;
    }
    
    /**
     * Ma se vypsat label
     * @deprecated css tridy a se nastavuji ve view
     * @return boolean
     */
    function getUseLabel() {
      return $this->useLabel;
    }

    /**
     * @deprecated  css tridy a se nastavuji ve view
     * @param boolean $useLabel
     * @return BaseField
     */
    function setUseLabel($useLabel) {
      $this->useLabel = $useLabel;
      return $this;
    }
    
    /**
     * pokud true tak se prazdny retezec meni na null
     * @param boolean $nullOnEmpty
     * @return BaseField
     */
    function setNullOnEmpty($nullOnEmpty) {
        $this->nullOnEmpty = $nullOnEmpty;
        return $this;
    }
    
    /**
     * @deprecated  use isOneWay instead
     * @return boolean
     */
    function isReadonly() {
        return $this->isOneWay();
    }

    /**
     * @deprecated  use setOneWay instead
     * @param boolean $readonly pokud je true, tak se nema hodnota z formulare zpet nastavovat do objektu
     */
    function setReadonly($readonly = true) {
        $this->setOneWay($readonly);
        return $this;
    }
    
    function isOneWay() {
        return $this->oneWay;
    }
    
    /**
     * @param boolean $oneWay pokud je true, tak se nema hodnota z formulare zpet nastavovat do objektu pomoci flush
     */
    function setOneWay($oneWay = true) {
        $this->oneWay = $oneWay;
        return $this;
    }
    
    
    public function __clone() {
        // naklonuje validacni pravidla
        $clonedRules = [];
        foreach($this->rules as $rule) {
            $clonedRule = clone $rule;
            $rule->setField($this);
            $clonedRules[] = $clonedRule;
        }
        $this->rules = $clonedRules;
    }
}
