<?php

namespace IZON\Forms\Fields;

/**
 * multiple checkboxes that return values based on which are checked
 */
class CheckboxListField extends BaseField {
    
    /**
     * @var array all options 
     */
    protected $options = [];
    
    /**
     * @var callable 
     */
    protected $keyLabelFunction = NULL;


    /**
     * 
     * @param string $name
     * @param string $label
     * @param array $params
     * @return PasswordField
     */
    public function __construct($name, $label = '', $params = []) {
        $this->name = $name;
        $this->label = $label;
        $this->propertyName = $this->name;
        $this->errors = [];
        return $this;
    }
    
    public function parse() {
        $selectedOptions = [];
        if( empty($this->value) ) {
            return $selectedOptions;
        }
        
        foreach($this->value as $name => $value ) {
            $selectedOptions[] = $this->options[$value]["option"];
        }
        return $selectedOptions;
    }

    public function serialize($values) {
        foreach($this->options as $key => $option) { // resetovat vsechny hodnoty
            $this->options[$key]["checked"] = false;
        }
        
        $keyLabelFunction = $this->keyLabelFunction;
        foreach($values as $index => $value) {
            $keyLabel = $keyLabelFunction($value, $index);
            $key = $keyLabel["key"];
            if( array_key_exists($key, $this->options) ) {
                $this->options[$key]["checked"] = true;
            }
        }
    }
    
    /**
     * sets value from request
     * @param array|null $value
     */
    public function setValue($value) {
        if( $value === null ) {
            $value = [];
        }
        parent::setValue($value);
        
        foreach($this->options as $key => $option) { // resetovat vsechny hodnoty
            $this->options[$key]["checked"] = false;
        }
        if( !empty($this->value) ) {
            foreach($this->value as $key ) { // set info which oprions are checked
                $this->options[$key]["checked"] = true;
            }
        }
    }

    /**
     * checks checkbox matching $value
     * does nothing if $value not present in options
     * @param mixed $value
     */
    public function checkOption($value) {
        $keyLabelFunction = $this->keyLabelFunction;

        // tries to find option by value equality, can fail for objects
        $index = 0;
        foreach($this->options as $option) {
            if( $option["option"] == $value ) {
                break;
            }
            $index++;
        }
        
        $keyLabel = $keyLabelFunction($value, $index);
        $key = $keyLabel["key"];
        if( array_key_exists($key, $this->options) ) {
            $this->options[$key]["checked"] = true;
        }
    }
    
    /**
     * unchecks checkbox matching $value
     * does nothing if $value not present in options
     * @param mixed $value
     */
    public function unCheckOption($value) {
        $keyLabelFunction = $this->keyLabelFunction;

        // tries to find option by value equality, can fail for objects
        $index = 0;
        foreach($this->options as $option) {
            if( $option["option"] == $value ) {
                break;
            }
            $index++;
        }
        
        $keyLabel = $keyLabelFunction($value, $index);
        $key = $keyLabel["key"];
        if( array_key_exists($key, $this->options) ) {
            $this->options[$key]["checked"] = false;
        }
    }


    /**
     * sets possible options
     * @param array $options array of options
     * @param callable $keyLableFunction function($option) that returns array ["key" => $uniqueIndetifierOfOption, "label" => $labelOfCheckbox] that uniqully indentify option
     */
    public function setOptions(array $options, $keyLabelFunction) {
        $this->keyLabelFunction = $keyLabelFunction;
        
        foreach($options as $index => $option) {
            $keyLabel = $keyLabelFunction($option, $index);
            if( !array_key_exists("key", $keyLabel) ) {
                throw new Exception("\$keyLableFunction doesn't return key for option");
            }
            if( !array_key_exists("label", $keyLabel) ) {
                throw new Exception("\$keyLableFunction doesn't return label for option");
            }
            
            $this->options[$keyLabel["key"]] = [
                "option" => $option,
                "label" => $keyLabel["label"],
                "checked" => false,
            ];
        }
    }
    
    function getOptions() {
        return $this->options;
    }
}
