<?php

namespace IZON\Forms;

use Exception;
use IZON\Logs\Logger;
use IZON\Forms\FormInterface;

/**
 * Skupina nekolika formuaru.
 * Je mozne vyuzit v nekolikastrankovych wizardech nebo vlozeni nekolika stejnych formularu najednou napriklad nekolik ckanku
 */
class FormSet implements FormInterface {
    /**
     * @var array prefixy u nadrazenych form setu 
     */
    protected $prependedPrefix = [];
    /**
     * prefix slouzi k zanoreni do pole predavaneho v set values a ke spravnemu vypisu
     * formName pro jednotliva pole ve formulari
     * @var array prefix pro vsechny pole formulare 
     */
    protected $prefix;
    /**
     * @var array, formulare, ktere se mohou v setu pobjevit jen 1x 
     */
    protected $forms;
    /**
     * @var array, formulare, ktere se mohou v setu pobjevit objevit vicekrat 
     */
    protected $dinamicForms;
    /**
     * validator pro cely formular
     * muze provadet validace zavisle na vice polich
     * @var callable 
     */
    protected $validator;
    /**
     * @var Logger 
     */
    protected $log;

    public function __construct(array $prefix = []) {
        $this->forms = [];
        $this->dinamicForms = [];

        $this->prefix = $prefix;

        $this->log = Logger::getLogger(__CLASS__);
    }

    /**
     * prida formular, ktery neni mozne vlozit vicekrat
     * @param FormInterface $form
     * @param array $prefix pripadny prefix pro dany formular, pokud ma furmular jiz nastaveny prefix tak ho prepise
     * @param array $params parametry pro praci s formularem
     */
    public function addForm(FormInterface $form, $objectClass, array $prefix = [], $params = []) {
        $prefixString = $this->serializePrefix($prefix);

        if($objectClass === NULL) {
            // TODO: vyresit pripad, pokud neni zadana trida
        } else if(!class_exists($objectClass)) {
            throw new Exception("Class $objectClass doesn't exist.");
        }

        $propertyName = NULL;
        if(isset($params["propertyName"])) {
            $propertyName = $params["propertyName"];
        } else if(count($prefix) > 0) {
            $propertyName = $prefix[0];
        }

        if(array_key_exists($prefixString, $this->forms)) {
            throw new Exception('Form with prefix ' . $prefixString . ' is already used.', 100);
        } else if(array_key_exists($prefixString, $this->dinamicForms)) {
            throw new Exception('DynamicForm with prefix ' . $prefixString . ' is already used.', 200);
        } else {
            $form->setPrefix(array_merge($this->getPrefix(), $prefix));
            if($form instanceof FormSet) {
                $form->setPrependedPrefix($this->prefix);
            }
            $this->forms[$prefixString] = [
                "form" => $form,
                "objectClass" => $objectClass,
                "propertyName" => $propertyName,
            ];
        }
    }

    /**
     * vratit formular podle prefixu, pokud neni prefix specializovan tak vraci formular, ktery byl pridan bez prefixu
     * 
     * @param array $prefix
     * @return FormInterface|Form|FormSet formular 
     */
    public function getForm(array $prefix = []) {
        $prefixString = $this->serializePrefix($prefix);

        if(!array_key_exists($prefixString, $this->forms)) {
            throw new Exception('Formular s prefixem  ' . $prefixString . ' neexistuje.');
        } else {
            return $this->forms[$prefixString]["form"];
        }
    }

    /**
     * @return FormInterface[] returns all frorms that are used only once
     */
    public function getForms() {
        $forms = [];
        foreach($this->forms as $prefix => $formInfo) {
            $forms[$prefix] = $formInfo["form"];
        }
        return $forms;
    }

    /**
     * prida formular, ktery je mozne vlozit vicekrat napriklad nekolik kategorii/autoru k jednomu clanku
     * 
     * $formFactory callback funkce je ve tvaru function($preffix, $formIndex)
     * $prefix, ktery se ma pro formular pouzivat
     * $formIndex index, kterym se odlisuji instance formularu
     * 
     * @param FormInterface $form form template to be cloned
     * @param array $prefix prefix, ktery se ma pouzit pred vsemi formulari
     * @param string $objectClass objekty jake tridy se maji doplnit pokud pocet prvku v property je mene nez formularu
     * @param array $params parametry minCount - minimalni pocet opakovani, maxCount - maximalni pocet opakovani, collectionFlushFunction = function(array $formsArray, $collection, $objectClass, $accessType) - funkce slouzisi ke zpracovani kolekce, da ze vyuzit, aby se obsahy formularu vlozily do spravnych objektu kolekce a doplnili se do dostatecneho poctu, musi bracet upravenou kolekci
     */
    public function addDinamicFormDefinition(FormInterface $form, $objectClass, array $prefix, $params = []) {
        $prefixString = $this->serializePrefix($prefix);

        if($objectClass === NULL) {
            // TODO: vyresit pripad, pokud neni zadana trida
        } else if(!class_exists($objectClass)) {
            throw new Exception("Class $objectClass doesn't exist.");
        }

        $propertyName = NULL;
        if(isset($params["propertyName"])) {
            $propertyName = $params["propertyName"];
        } else if(count($prefix) > 0) {
            $propertyName = $prefix[0];
        }

        if(array_key_exists($prefixString, $this->forms)) {
            throw new \Exception('Form with prefix ' . $prefixString . ' is already used.', 100);
        } else if(array_key_exists($prefixString, $this->dinamicForms)) {
            throw new \Exception('DynamicForm with prefix ' . $prefixString . ' is already used.', 200);
        } else {
//            var_dump(get_class($form));
//            var_dump($prefix);
            $form->setPrefix(array_merge($this->getPrefix(), $prefix));
            if($form instanceof FormSet) {
                $form->setPrependedPrefix($this->prefix);
            }
            $this->dinamicForms[$prefixString] = [
                "formDefinition" => $form, // contains template of form
                "objectClass" => $objectClass,
                "params" => $params, // 
                "forms" => [], // actual forms with data
                "propertyName" => $propertyName,
            ];
//            var_dump($prefix);
//            var_dump($form->getPrefix());
        }
    }

    /**
     * @param array $prefix vrati formular odpovidejici definici dinamickych formularu
     * @return FormInterface definition of dinamic from
     */
    public function getDinamicFormDefinition(array $prefix) {
        $prefixString = $this->serializePrefix($prefix);

        if(array_key_exists($prefixString, $this->dinamicForms)) {
            return $this->dinamicForms[$prefixString]["formDefinition"];
        } else {
            throw new \Exception('DynamicForm with prefix ' . $prefixString . ' doesn\'t exist.', 100);
        }
    }

    /**
     * @param array $prefix vrati formular odpovidejici definici dinamickych formularu
     */
    public function getDinamicFormDefinitions() {
        $dinamicFormsDefinitions = [];
        foreach($this->dinamicForms as $prefix => $dinFormInfo) {
            $dinamicFormsDefinitions[$prefix] = $dinFormInfo["formDefinition"];
        }
        return $dinamicFormsDefinitions;
    }

    /**
     * vytvori novy dinamicly formular ulozeny pod prefiex ktery bude mit prefix - array_mege($prefix, [$index])
     * da se pouzit pro pridavani novych formularu do stranky
     * @param array $prefix
     * @param serializable $index
     */
    public function createDinamicForm(array $prefix, $index) {
        $prefixString = $this->serializePrefix($prefix);

        if(array_key_exists($prefixString, $this->dinamicForms)) {
            $formDef = $this->dinamicForms[$prefixString]["formDefinition"];
            $form = clone $formDef;
            $form->setPrefix(array_merge($prefix, [$index]));
            return $form;
        } else {
            throw new \Exception('DynamicForm with prefix ' . $prefixString . ' doesn\'t exist.', 100);
        }
    }

    /**
     * vraci pocet dinamickych formularu
     * pokud neni zadan $prefix vraci se formulade pridane pomoci addDinamicForm be prefixu
     * @param array $prefix vrati pole vsech dinamickych formularu pod danym preffixem
     */
    public function getDinamicFormsCount(array $prefix) {
        $prefixString = $this->serializePrefix($prefix);

        if(array_key_exists($prefixString, $this->dinamicForms)) {
            return count($this->dinamicForms[$prefixString]["forms"]);
        } else {
            throw new \Exception('DynamicForm with prefix ' . $prefixString . ' doesn\'t exist.', 100);
        }
    }

    /**
     * @param array $prefix vrati pole vsech dinamickych formularu 
     */
    public function getDinamicForms(array $prefix) {
        $prefixString = $this->serializePrefix($prefix);

        if(array_key_exists($prefixString, $this->dinamicForms)) {
            return $this->dinamicForms[$prefixString]["forms"];
        } else {
            throw new \Exception('DynamicForm with prefix ' . $prefixString . ' doesn\'t exist.', 100);
        }
    }

    /**
     * sets validator for whole SormSet
     * @param callable $validator is a method with one parame FormSet $formSet, it sets found errors tirectly to $form
     */
    function setValidator($validator) {
        if(!is_callable($validator)) {
            throw new Exception("Validator must be callable");
        }
        $this->validator = $validator;
    }

    /**
     * provede validaci FormSet jako celku
     */
    public function validate() {
        // validuje normalni formulare
        foreach($this->forms as $form) {
            $form["form"]->validate();
        }
        // validuje dinamicke formulare
        foreach($this->dinamicForms as $dinamicForm) {
            foreach($dinamicForm["forms"] as $form) {
                $form->validate();
            }
        }
        // vola validator na cely FormSet
        if($this->validator) {
            $validator = $this->validator;
            $validator($this);
        }
    }

    /**
     * zjisti jestli ma formular nebo nejake formularove pole chybu
     * @return boolean
     */
    public function hasErrors() {
        $hasErrors = false;
        foreach($this->forms as $form) {
            $hasErrors |= $form["form"]->hasErrors();
        }
        // validuje dinamicke formulare
        foreach($this->dinamicForms as $dinamicForm) {
            foreach($dinamicForm["forms"] as $form) {
                $hasErrors |= $form->hasErrors();
            }
        }
        return $hasErrors || !empty($this->errors);
    }

    public function hasFormSetErrors() {
        return !empty($this->errors);
    }

    /**
     * prida chybu navazanou primo na formulari ne navazane na formularove pole
     * @param string $errorMessage
     */
    function addFormSetError($errorMessage) {
        $this->errors[] = $errorMessage;
    }

    /**
     * vrati chyby navazane primo na formulari ne navazane na formularove pole
     * @return array
     */
    function getFormSetErrors() {
        return $this->errors;
    }

    /**
     * @return array 
     * TODO: doimplementovat
     */
//    function getErrorsArray() {
//        $errors = [];
//        foreach ($this->getFileds() as $fild) {
//            if ($fild->hasErrors()) {
//                $errorsArray = $fild->getErrors();
//                $errors[] = ["formName" => $fild->getFormName(), "errors" => $errorsArray];
//            }
//        }
//        return $errors;
//    }

    /**
     * inicializuje jednotlive podformulare
     * @param array $array
     */
    public function setValues($array) {
        // spracuje normalni formulare
        foreach($this->forms as $prefixString => $formInfo) {
            $form = $formInfo["form"];
            $same = $this->getSamePrefix($form->getPrefix());
            $prefix = array_merge($this->prefix, array_slice($form->getPrefix(), count($same)));
            $form->setPrefix($prefix);
            $form->setValues($array);
        }
        // spracuje dinamicke formulare
        foreach($this->dinamicForms as $prefixString => $dinForms) {
            $formDef = $dinForms["formDefinition"];
            $same = $this->getSamePrefix($formDef->getPrefix());
            $prefix = array_merge($this->prefix, array_slice($formDef->getPrefix(), count($same)));
            $prefixedArray = $array;
            foreach($prefix as $pref) { // presun se na prefix
                if(isset($prefixedArray[$pref])) {
                    $prefixedArray = $prefixedArray[$pref];
                } else {
                    $prefixedArray = [];
                }
            }
            foreach($prefixedArray as $index => $valuesArray) {
                $form = clone $formDef;
                $form->setPrefix(array_merge($prefix, [$index]));
                $form->setValues($array);
                $this->dinamicForms[$prefixString]["forms"][$index] = $form;
            }
        }
    }

    /**
     * @return array reprezentuje preffix
     */
    public function getPrefix() {
        return $this->prefix;
    }

    public function setPrefix(array $prefix) {
        $this->prefix = $prefix;
        $this->prependPrefixToSubForms(array_merge($prefix), true);
    }

    function setPrependedPrefix($prependedPrefix) {
        $this->prefix = array_merge($prependedPrefix, array_splice($this->prefix,count($this->prependedPrefix)));
        $this->prependedPrefix = $prependedPrefix;
        foreach($this->forms as $prefixString => $formInfo) {
            $form = $formInfo["form"];
            if($form instanceof FormSet) {
                $form->setPrependedPrefix($this->prefix);
            }
        }
        // spracuje dinamicke formulare
        foreach($this->dinamicForms as $prefixString => $dinForms) {
            $formDef = $dinForms["formDefinition"];
            if($formDef instanceof FormSet) {
                $formDef->setPrependedPrefix($this->prefix);
            }
        }
    }

    protected function getSamePrefix(array $prefix) {
        $same = [];
        $max = min(count($prefix), count($this->getPrefix()));
        for($i = 0; $i < $max; $i++) {
            if($prefix[$i] === $this->getPrefix()[$i]) {
                $same[] = $prefix[$i];
            }
        }
        return $same;
    }

    protected function prependPrefixToSubForms(array $prefix, $removeSamePrefix = false) {
        foreach($this->forms as $prefixString => $formInfo) {
            $form = $formInfo["form"];
            if($removeSamePrefix) {
                $same = $this->getSamePrefix($form->getPrefix());
                $form->setPrefix(array_merge($prefix, array_slice($form->getPrefix(), count($same))));
            } else {
                $form->setPrefix(array_merge($prefix, $form->getPrefix()));
            }

//            }
        }
        // spracuje dinamicke formulare
        foreach($this->dinamicForms as $prefixString => $dinForms) {
            $formDef = $dinForms["formDefinition"];
//            if( $formDef instanceof FormSet ) {
//                $formDef->prependPrefixToSubForms( array_merge($prependedPrefix, $form->getPrefix()) );
//            } else {

            if($removeSamePrefix) {
                $same = $this->getSamePrefix($formDef->getPrefix());
                $formDef->setPrefix(array_merge($prefix, array_slice($formDef->getPrefix(), count($same))));
            } else {
                $formDef->setPrefix(array_merge($prefix, $formDef->getPrefix()));
            }
//            }
        }
    }

    /**
     * ze zadaneho objektu a podpobjektu nastavi hodnoty jednotlivych formularu
     * @param object $object
     * @param integer $accessType how to access properties of object EDIT_FLUSH_ACCESS_GETTERS_SETTERS - through getters/setters , EDIT_FLUSH_ACCESS_PROPERTIES - directly to propertis
     */
    public function edit($object, $accessType = Form::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) {
        if(is_null($object)) {
            throw new Exception("Objekt ze ktereho se data kopiruji nesmi byt null");
        }

        /* @var $reflectionClass \ReflectionClass */
        $reflectionClass = new \ReflectionClass($object);

        // projit normalni formulare
        foreach($this->forms as $formInfo) {
            /* @var $form Form */
            $form = $formInfo['form'];
            $propertyName = $formInfo["propertyName"];
            if($propertyName == NULL) { // mapuje se primo na  $object
                $form->edit($object, $accessType);
                continue;
            }

            if($accessType == Form::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
                if($reflectionClass->hasProperty($propertyName)) {
                    $reflectionProperty = $reflectionClass->getProperty($propertyName);
                    $reflectionProperty->setAccessible(true);

                    $subObject = $reflectionProperty->getValue($object);
                    if($subObject != NULL) {
                        $form->edit($subObject);
                    }
                } else if($reflectionClass->hasMethod("__isset") // ma metodu isset
                        && $object->__isset($propertyName)) { // ma property nadefinovane pomoci __isset, __get a __set metod
                    $subObject = $object->$propertyName;
                    if($subObject != NULL) {
                        $form->edit($subObject);
                    }
                } else {
                    $this->log->info("Doesn't have property: " . $propertyName . " to edit data from");
                }
            } else if($accessType == Form::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
                $getterName = \IZON\Object\createGetterMethodName($propertyName);
                if($reflectionClass->hasMethod($getterName)) {
                    $getterMethod = $reflectionClass->getMethod($getterName);

                    $subObject = $getterMethod->invoke($object);
                    if($subObject != NULL) {
                        $form->edit($subObject);
                    }
                } else {
                    $this->log->info("Doesn't have getter: " . $getterName . " to edit data from");
                }
            } else {
                throw new Exception("Unsupported access type " . $accessType);
            }
        }

        // projit dinamicke formulare, musi se mapovat na array nebo na kolekci
        foreach($this->dinamicForms as $dinaFormIndex => $dinaFormInfo) {

            /* @var $formDefinition Form */
            $formDefinition = $dinaFormInfo['formDefinition'];
            $propertyName = $dinaFormInfo["propertyName"];

            if($accessType == Form::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
                if($reflectionClass->hasProperty($propertyName)) {
                    $reflectionProperty = $reflectionClass->getProperty($propertyName);
                    $reflectionProperty->setAccessible(true);

                    $collection = $reflectionProperty->getValue($object);
                    foreach($collection as $index => $element) { // pro vsechny emementy vytvor pole
                        $clonedForm = clone $formDefinition;
//                        if( $clonedForm instanceof FormSet ) {
//                            $clonedForm->prependPrefixToSubForms( array_merge($formDefinition->getPrefix(), [$index]) );
//                        } else {
                        $clonedForm->setPrefix(array_merge($formDefinition->getPrefix(), [$index]));
//                        }
                        $clonedForm->edit($element, $accessType);
                        $this->dinamicForms[$dinaFormIndex]["forms"][$index] = $clonedForm;
                    }
                } else if($reflectionClass->hasMethod("__isset") // ma metodu isset
                        && $object->__isset($propertyName)) { // ma property nadefinovane pomoci __isset, __get a __set metod
                    $collection = $object->$propertyName;
                    foreach($collection as $index => $element) { // pro vsechny emementy vytvor pole
                        $clonedForm = clone $formDefinition;
                        $clonedForm->setPrefix(array_merge($formDefinition->getPrefix(), [$index]));
                        $clonedForm->edit($element, $accessType);
                        $this->dinamicForms[$dinaFormIndex]["forms"][$index] = $clonedForm;
                    }
                } else {
                    $this->log->info("Doesn't have property: " . $propertyName . " to edit data from");
                }
            } else if($accessType == Form::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
                $getterName = \IZON\Object\createGetterMethodName($propertyName);
                if($reflectionClass->hasMethod($getterName)) {
                    $getterMethod = $reflectionClass->getMethod($getterName);

                    $collection = $getterMethod->invoke($object);
                    foreach($collection as $index => $element) { // pro vsechny emementy vytvor pole
                        $clonedForm = clone $formDefinition;
//                         else {
                        $clonedForm->setPrefix(array_merge($formDefinition->getPrefix(), [$index]));
//                        }
                        if($clonedForm instanceof FormSet) {
                            $clonedForm->prependPrefixToSubForms(array_merge($formDefinition->getPrefix(), [$index]), true);
                        }
                        $this->dinamicForms[$dinaFormIndex]["forms"][$index] = $clonedForm;

                        $clonedForm->edit($element, $accessType);
                    }
                } else {
                    $this->log->info("Doesn't have getter : " . $getterName . " to fedit data from");
                }
            } else {
                throw new Exception("Unsupported access type " . $accessType);
            }
        }
    }

    /**
     * z hodnot ve formularich nastavi hodnoty properit v objektu a podobjektech
     * @param object $object
     * @param integer $accessType how to access properties of object EDIT_FLUSH_ACCESS_GETTERS_SETTERS - through getters/setters , EDIT_FLUSH_ACCESS_PROPERTIES - directly to propertis
     */
    public function flush($object, $accessType = Form::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) {
        if(is_null($object)) {
            throw new Exception("Objekt do ktereho se data kopiruji nesmi byt null");
        }

        /* @var $reflectionClass \ReflectionClass */
        $reflectionClass = new \ReflectionClass($object);

        // projit normalni formulare
        foreach($this->forms as $formInfo) {
            /* @var $form Form */
            $form = $formInfo['form'];
            $propertyName = $formInfo["propertyName"];
            if($propertyName == NULL) { // mapuje se primo na  $object
                $form->flush($object, $accessType);
                continue;
            }

            if($accessType == Form::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
                if($reflectionClass->hasProperty($propertyName)) {
                    $reflectionProperty = $reflectionClass->getProperty($propertyName);
                    $reflectionProperty->setAccessible(true);

                    $subObject = $reflectionProperty->getValue($object);
                    if($subObject == NULL) { // pokud je objekt null
                        $objectClass = $formInfo["objectClass"]; // objekty jake tridy se ma doplnovat
                        $subObject = new $objectClass();
                    }
                    $form->flush($subObject);
                    $reflectionProperty->setValue($object, $subObject);
                } else if($reflectionClass->hasMethod("__isset") // ma metodu isset
                        && $object->__isset($propertyName)) { // ma property nadefinovane pomoci __isset, __get a __set metod
                    $subObject = $object->$propertyName;
                    if($subObject == NULL) {
                        $objectClass = $formInfo["objectClass"]; // objekty jake tridy se ma doplnovat
                        $subObject = new $objectClass();
                    }
                    $form->flush($subObject);
                    $object->$propertyName = $subObject;
                } else {
                    $this->log->info("Doesn't have property: " . $propertyName . " to flush data to");
                }
            } else if($accessType == Form::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
                $getterName = \IZON\Object\createGetterMethodName($propertyName);
                $setterName = \IZON\Object\createSetterMethodName($propertyName);
                if($reflectionClass->hasMethod($getterName) // mame getter
                        && $reflectionClass->hasMethod($setterName)) { // mame setter
                    $getterMethod = $reflectionClass->getMethod($getterName);
                    $setterMethod = $reflectionClass->getMethod($setterName);

                    $subObject = $getterMethod->invoke($object);
                    if($subObject == NULL) { // pokud je objekt null
                        $objectClass = $formInfo["objectClass"]; // objekty jake tridy se ma doplnovat
                        $subObject = new $objectClass();
                    }

                    $form->flush($subObject);
                    // nastavi kolekci zpet do objektu pomoci setteru
                    $setterMethod->invoke($object, $subObject);
                } else {
                    $this->log->info("Doesn't have getter: " . $getterName . " and setter: " . $setterName . " to flush data to");
                }
            } else {
                throw new Exception("Unsupported access type " . $accessType);
            }
        }

        // projit dinamicke formulare, musi se mapovat na array nebo na kolekci
        foreach($this->dinamicForms as $dinaFormIndex => $dinaFormInfo) {
            /* @var $formDefinition Form */
            $forms = $dinaFormInfo['forms'];
            $propertyName = $dinaFormInfo["propertyName"];

            if($accessType == Form::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
                if($reflectionClass->hasProperty($propertyName)) {
                    $reflectionProperty = $reflectionClass->getProperty($propertyName);
                    $reflectionProperty->setAccessible(true);

                    $collection = $reflectionProperty->getValue($object);
                    $collection = $this->handleFlushCollection($forms, $collection, $dinaFormInfo, $accessType);
                    $reflectionProperty->setValue($object, $collection);
                } else if($reflectionClass->hasMethod("__isset") // ma metodu isset
                        && $object->__isset($propertyName)) { // ma property nadefinovane pomoci __isset, __get a __set metod
                    $collection = $object->$propertyName;
                    $collection = $this->handleFlushCollection($forms, $collection, $dinaFormInfo, $accessType);
                    $object->$propertyName = $collection;
                } else {
                    $this->log->info("Doesn't have property: " . $propertyName . " to flush data to");
                }
            } else if($accessType == Form::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
                $getterName = \IZON\Object\createGetterMethodName($propertyName);
                $setterName = \IZON\Object\createSetterMethodName($propertyName);
                if($reflectionClass->hasMethod($getterName) // mame getter
                        && $reflectionClass->hasMethod($setterName)) { // mame setter
                    $getterMethod = $reflectionClass->getMethod($getterName);
                    $setterMethod = $reflectionClass->getMethod($setterName);
                    $collection = $getterMethod->invoke($object); // ziskat kolekci objektu

                    $collection = $this->handleFlushCollection($forms, $collection, $dinaFormInfo, $accessType);

                    // nastavi kolekci zpet do objektu pomoci setteru
                    $setterMethod->invoke($object, $collection);
                }
            } else {
                throw new Exception("Unsupported access type " . $accessType);
            }
        }
    }

    /**
     * serializuje prefix do stringu, pod kterym se uklada do vnitrnih promennych setu
     * @param array $prefix
     * @return string
     */
    protected function serializePrefix(array $prefix) {
        $preffixString = '';
        if($prefix != NULL) {
            $preffixString = "[" . implode(",", $prefix) . "]";
        }
        return $preffixString;
    }

    /**
     * 
     * @param array[Form] $forms
     * @param array $collection
     * @param array $dinaFormInfo
     * @return array
     * @throws Exception
     */
    protected function handleFlushCollection($forms, $collection, $dinaFormInfo, $accessType) {
        $objectClass = $dinaFormInfo["objectClass"]; // objekty jake tridy se ma doplnovat

        if(isset($dinaFormInfo["params"]["collectionFlushFunction"])) { // ma nastavenou funkci pro zpracovani kolekce
            $callback = $dinaFormInfo["params"]["collectionFlushFunction"];
            if(!is_callable($callback)) { // collectionFlushCallback neni funkce
                throw new Exception("collectionFlushFunction must be callable.");
            }

            // zpracuje kolekci podle vstupu z formularu
            $collection = $callback($forms, $collection, $objectClass, $accessType);

            if(!is_array($collection)) {
                throw new Exception("collectionFlushFunction must return collection/array.");
            }
        } else {
            $formsArrayKeys = array_keys($forms);
            if(is_array($collection)) {
                $collectionArrayKeys = array_keys($collection);
            } else {
                $collectionArrayKeys = [];
                foreach($collection as $key => $value) {
                    $collectionArrayKeys[] = $key;
                }
            }

            // smaž z kolekce co není ve formuláři
            $keysDelete = array_diff($collectionArrayKeys, $formsArrayKeys);
            foreach($keysDelete as $key) {
                unset($collection[$key]);
            }
            // projdi formulare a vyber z kolekce
            foreach($formsArrayKeys as $index) {
                // jaky furmular se ma flushovat
                $form = $forms[$index];
                if(!isset($collection[$index])) { // v kolekci neexituje objekt
                    $element = new $objectClass();
                } else {
                    $element = $collection[$index];
                    unset($collection[$index]);
                }
                // flushnout data do objektu
                $form->flush($element, $accessType);
                $collection[$index] = $element;
            }
        }

        return $collection;
    }

    public function __clone() {
        // spracuje normalni formulare
        foreach($this->forms as $prefixString => $formInfo) {
            $formDef = $formInfo["form"];
            $this->forms[$prefixString]["form"] = clone $formDef;
        }
        // spracuje dinamicke formulare
        foreach($this->dinamicForms as $prefixString => $dinForms) {
            $formDef = $dinForms["formDefinition"];
            $this->dinamicForms[$prefixString]["formDefinition"] = clone $formDef;
        }
    }

}
