<?php

namespace IZON\Forms\Fields;

use IZON\Forms\ValidationRule;

/**
 * Pole pro telefoni cislo
 */
class PhoneField extends CharField {

    /**
     * @var string regex for number with international prefix
     */
    protected $numberWithCountryCodeRegex = "/^(\+|00)[1-9][0-9]{0,4}[0-9]{9,15}$/";

    /**
     * @var string regex for number without international prefix
     */
    protected $numberWithoutCountryCodeRegex = "/^[0-9]{9,15}$/";


    public function __construct($name, $label = '', $params = array()) {
        parent::__construct($name, $label, $params);

        $numberWithCountryCodeRegex = $this->numberWithCountryCodeRegex;
        $numberWithoutCountryCodeRegex = $this->numberWithoutCountryCodeRegex;

        $rule = new ValidationRule(
            function(BaseField $field, $args = [])
                use ($numberWithCountryCodeRegex, $numberWithoutCountryCodeRegex) {
                if($field->parse() == '') {
                    return true;
                }

                $value = preg_replace('/\s+/', '', $field->parse());
                return preg_match($numberWithCountryCodeRegex, $value) // phone number with country code
                    || preg_match($numberWithoutCountryCodeRegex, $value); // phone number without country code
            },
            _('Zadané telefoní číslo není platné.')
        );
        $this->addRule($rule);

        return $this;
    }

    public function removeAllRules() {
        $this->rules = [];
    }

    public function setNumberWithCountryCodeRegex(string $numberWithCountryCodeRegex): void {
        $this->numberWithCountryCodeRegex = $numberWithCountryCodeRegex;
    }

    public function setNumberWithoutCountryCodeRegex(string $numberWithoutCountryCodeRegex): void {
        $this->numberWithoutCountryCodeRegex = $numberWithoutCountryCodeRegex;
    }
}
