<?php
namespace IZON\Forms;

/**
 * Vygenerovani formularoveho pole na zaklade templatu vzhledu
 *
 * @author Vitezslav Jahn <jahn@izon.cz>
 */
class FormFieldTemplate {
  const TYPE_BACKEND                    = 'admin';
  const TYPE_FRONTEND                   = 'web';
  const TYPE_FRONTEND_ADMIN             = 'web-admin';
  const LABEL_TEXT_BEFORE_FIELD         = 'before';
  const LABEL_TEXT_AFTER_FIELD          = 'after';


  private function __construct() {
    
  }

  /**
   * 
   * @param string $type admin|web|web-admin
   * @param string $file template file name
   * @return array
   * @throws \Exception
   */
  protected static function loadTemplate($type, $file) {
    $path = __BASE_DIR__.'/app/static/'.$type.'/form-flelds-templates/'.$file.'.json';

    if(file_exists($path)) {
      $_data = file_get_contents ($path);
      return json_decode($_data, true);
    }

    if($type == self::TYPE_BACKEND) {
      $path = __BASE_DIR__.'/vendor/izon/admin-static/static/form-flelds-templates/'.$file.'.json';
      if(file_exists($path)) {
        $_data = file_get_contents ($path);
        return json_decode($_data, true);
      } 
    }
    
    throw new \Exception('Soubor s definici vzhledu formularovych poli - '.$path.' - nebyl nalezen.');
  }
  
  /**
   * Return HTML code for form filed
   * 
   * @param string $templateFile rendering template name
   * @param string $type admin|web|web-admin
   * @param \IZON\Forms\Fields\BaseField $field
   * @param array $params array of parameters
   * @return string
   */
  public static function getFiled($templateFile, $type, Fields\BaseField $field, $params = array()) {
    $template = self::loadTemplate($type, $templateFile);
    switch (get_class($field)) {
      case Fields\CharField::class:   
        $type = 'text'; 
        $input  = self::parseInput($type, $template[$type]['field'], $field, $params['field']);
        break;
      case Fields\DateFiled::class:         
      case Fields\DateTimeFiled::class:     
        $type = 'date'; 
        if(Fields\DateTimeFiled::class == get_class($field)) {
          $type = 'datetime';
        }
        $input  = self::parseInput('text', $template[$type]['field'], $field, $params['field']);
        break;
      case Fields\CheckboxField::class:      
        $type = 'checkbox'; 
        $input  = self::parseInput($type, $template[$type]['field'], $field, $params['field']);
        break;
      case Fields\EmailField::class:      
        $type = 'text'; 
        $input  = self::parseInput('email', $template[$type]['field'], $field, $params['field']);
        break;
      case Fields\PhoneField::class:      
        $type = 'text'; 
        $input  = self::parseInput('tel', $template[$type]['field'], $field, $params['field']);
        break;
      case Fields\FloatField::class: 
      case Fields\IntegerField::class:
      case Fields\IntegerRangeField::class:
      case Fields\FloatRangeField::class:     
        $type = 'text'; 
        $input  = self::parseInput('number', $template[$type]['field'], $field, $params['field']);
        break;
      case Fields\PasswordField::class:
        $type = 'password'; 
        $input  = self::parseInput('password', $template[$type]['field'], $field, $params['field']);
        break;
      case Fields\SelectField::class:        
        $type = 'select'; 
        $input  = self::parseSelect($template[$type]['field'], $field, $params['field']);
        break;
      case Fields\RadioListField::class:        
        $type = 'radio'; 
        $input  = self::parseRadio($template[$type]['field'], $field, $params['field']);
        break;
      case Fields\CheckboxListField::class:
        $fieldType = 'checkbox';
        $type = 'checkboxList';
        $input  = self::parseCheckboxListField($template[$type]['field'], $field, $params['field']);
        break;
      case Fields\TextField::class:
        $type = 'textarea'; 
        $input  = self::parseTextArea($template[$type]['field'], $field, $params['field']);
        break;
      case Fields\IdField::class:
      case Fields\HiddenField::class:
        $attrs  = [];
        foreach ($field->getAttributes() as $key => $value) {
            $attrs[] = $key.'="'.$value.'"';
        }
        if( array_key_exists('field', $params)
            && array_key_exists('attributes', $params['field']) ) {
            foreach($params['field']["attributes"] as $key => $value) {
                $attrs[] = $key.'="'.$value.'"';
            }
        }
        $input = '<input type="hidden" value="'.$field->getValue().'" name="'.$field->getFormName().'"'.(!empty($attrs) ? ' '.implode(' ', $attrs) : '').'>';
        $params['raw'] = true;
      default:
        break;
    }
    
    // chceme jen cisty input
    if($params['raw']) {
      return $input;
    }
    // WRAPPER
    $wrapperSettings = (array)$template[$type]['wrapper'];
    if($field->hasErrors()) {
      if(array_key_exists('attributes', $wrapperSettings)) {
        $wrapperSettings['attributes']['class'] .= ' error';
      } else {
        $wrapperSettings['attributes']['class'] = 'error';
      }
    }
    if(!array_key_exists('captionPosition', $wrapperSettings)) {
      $wrapperSettings['captionPosition'] = self::LABEL_TEXT_BEFORE_FIELD;
    }
    if(array_key_exists('captionPosition', (array)$params['wrapper']) && !empty($params['wrapper']['captionPosition'])) {
      $wrapperSettings['captionPosition'] = $params['wrapper']['captionPosition'];
    }
    $wrapper  = self::generateWrapper($wrapperSettings, (array)$params['wrapper']);
    
    
    // CAPTION
    $caption = '';
    if(!$params['noCaption']) {
      $captionText = $params['caption']['text'];
      if(empty($captionText)) {
        $captionText = $field->getLabel();
      }
      $caption = self::generateCaption($template[$type]['caption'], $captionText, (array)$params['caption']);
    }
       
    $error  = $field->hasErrors() ? self::generateFieldError($template[$type]['error'], $field->getErrors(), $params['error']) : '';
        
    $content = '';
    switch ($wrapperSettings['captionPosition']) {
      case self::LABEL_TEXT_BEFORE_FIELD:
        $content = $caption . $input . $error ."\n";
        break;
      case self::LABEL_TEXT_AFTER_FIELD:
        $content = $input . $error . $caption ."\n";
        break;
    }
    
    return str_replace('__CONTENT__', $content, $wrapper);
  }
  
  /**
   * 
   * @param array $settings
   * @param array $defaults
   * @return array
   */
  protected static function setDefaults($settings, $defaults) {
    if(!empty($defaults) && is_array($defaults)) {
      foreach ($defaults as $key => $value) {
        if(!array_key_exists($key, (array)$settings)) {
           $settings[$key] = $value;
        }
      }
    }
    return $settings;
  }
  /**
   * Aplikuje defaultni hodnoty a zmeny predane pres parametr vykreslovaci fce oproti vzoru
   * 
   * @param array $settings
   * @param array $params
   * @param array $defaults
   * @return string
   */
  protected static function applyPatternChanges($settings, $params, $defaults = []) {
    $settings = self::setDefaults($settings, $defaults);
    if(!empty($params) && is_array($params)) {
      foreach ($params as $key => $value) {
        if($key == 'attributes')          continue;
        if(array_key_exists($key, (array)$settings) && $settings[$key] != $params[$key]) {
          $settings[$key] = $params[$key];
        }
      }
    }
    return $settings;
  }
  
  /**
   * 
   * @param array $attrs
   * @param array $params
   * @return string
   */
  protected static function generateWrapper($settings, $params = array()) {
    $settings = self::applyPatternChanges($settings, $params, array('tag' => 'label', 'captionPosition' => self::LABEL_TEXT_BEFORE_FIELD));
    $attrs    = self::mergeAttributes($settings['attributes'], $params['attributes']);
    $str      = [];
    if(!empty($attrs) && is_array($attrs)) {
      foreach ($attrs as $key => $value) {
        $str[] = $key.'="'.$value.'"';
      }
    }
    $ret = '<'.$settings['tag'].' '.  implode(' ', $str).'>__CONTENT__</'.$settings['tag'].'>';
    return $ret;
  }
  /**
   * 
   * @param array $settings
   * @param string $text
   * @param array $params
   * @return string
   */
  protected static function generateCaption($settings, $text, $params = array()) {
    $settings = self::applyPatternChanges($settings, $params, array('tag' => 'span'));
    $settings['attributes'] = self::setDefaults($settings['attributes'], array('class'=>'form__item__label'));
    
    $attrs = self::mergeAttributes($settings['attributes'], $params['attributes']);
    $str    = [];
    if(!empty($attrs) && is_array($attrs)) {
      foreach ($attrs as $key => $value) {
        $str[] = $key.'="'.$value.'"';
      }
    }
    
    return '<'.$settings['tag'].' '.implode(' ', $str).'>'.$text.'</'.$settings['tag'].'>';
  }
  
  protected static function generateFieldError( $settings, $errors = [], $params = array()) {
    $settings = self::applyPatternChanges($settings, $params, array('tag' => 'div', 'separator' => '<br>'));
    $settings['attributes'] = self::setDefaults($settings['attributes'], array('class'=>'message'));
    
    $attrs = self::mergeAttributes($settings['attributes'], $params['attributes']);
    
    $str    = [];
    if(!empty($attrs) && is_array($attrs)) {
      foreach ($attrs as $key => $value) {
        $str[] = $key.'="'.$value.'"';
      }
    }
    return '<'.$settings['tag'].' '.  implode(' ', $str).'>'.implode($settings['separator'], $errors).'</'.$settings['tag'].'>';
  }
  /**
   * 
   * @param string $type [text|checkbox|number|password]
   * @param array $pattern
   * @param \IZON\Forms\Fields\BaseField $field
   * @param array $params
   * @return string
   */
  protected static function parseInput($type, $pattern, Fields\BaseField $field, $params = array()) {
    $attrs           = $pattern['attributes'];
    if(!in_array($type, ['password'])) {
      
      if($type == 'checkbox') {
        if(!empty($field->getValue())) {
          $attrs['value'] = self::clearString($field->getValue());
        }
      } else {
        $attrs['value'] = self::clearString($field->getValue());
      }
      
    } 
    if($type == 'checkbox' && $field->getValue() == true) {
      $attrs['checked'] = 'checked';
    }
    if($type == 'number' && $field->getStep() + 0 > 0) {
      $attrs['step'] = str_replace(',', '.', $field->getStep());
      $min = $field->getMinValue();
      if(isset($min)) {
        $attrs['min'] = str_replace(',', '.', $min);
      }
      $max = $field->getMaxValue();
      if(isset($max)) {
        $attrs['max'] = str_replace(',', '.', $max);
      }
    }
    $attrs = self::getAttrs($attrs, $field, $params['attributes']);
    
    return '<input type="'.$type.'" '.  implode(' ', $attrs).'/>'. "\n";
  }
  
  /**
   * 
   * @param array $pattern
   * @param \IZON\Forms\Fields\BaseField $field
   * @param array $params
   * @return string
   */
  protected static function parseTextArea($pattern, Fields\BaseField $field, $params = array()) {
    
    /*if($field->hasAttribute('rows')) {
      $pattern['attributes']['rows'] = $field->getAttribute('rows');
    }
    if($field->hasAttribute('cols')) {
      $pattern['attributes']['cols'] = $field->getAttribute('cols');
    }
    if($field->hasAttribute('placeholder')) {
      $pattern['attributes']['placeholder'] = $field->getAttribute('placeholder');
    }*/
    $attrs = self::getAttrs($pattern['attributes'], $field, $params['attributes']);
    /*if(!array_key_exists('rows', $params)) {
      $attrs[] = 'rows="'.($field->hasAttribute("rows") && $field->getAttribute('row') + 0 > 0 ? $field->getAttribute('row') : 5).'"';
    }*/

    if($field->hasAttribute('readonly')) {
      $attrs['readonly'] = 'readonly';
    }
    if($field->hasAttribute('disabled')) {
      $attrs['disabled'] = 'disabled';
    }
    
    return  '<textarea '.  implode(' ', $attrs).'>'. clearString($field->getValue()) .'</textarea>'. "\n";
  }
  
  /**
   * 
   * @param array $pattern
   * @param \IZON\Forms\Fields\BaseField $field
   * @param array $params
   * @return string
   */
  protected static function parseSelect($pattern, Fields\BaseField $field, $params = array()) {
    
    $attrs = self::getAttrs($pattern['attributes'], $field, $params['attributes']);
    foreach($field->getOptions() as $key => $value) {
        if( is_array($value) ) {
            $options .= '<optgroup label="'. $value["label"] .'">';
            
            foreach($value["options"] as $optVal => $optLable) {
                $options .= '<option value="'. $optVal .'"'. ((string)$optVal == (string)$field->getValue() ? ' selected="selected"' : '') .'>'
                            . $optLable
                            .'</option>';
            }
            
            $options .= "</optgroup>";
        } else {
            $options .= '<option value="'. $key .'"'. ((string)$key == (string)$field->getValue() ? ' selected="selected"' : '') .'>'
                        . $value
                        .'</option>';
        }
    }
    if($field->hasAttribute('readonly')) {
      $attrs['readonly'] = 'readonly';
    }
    if($field->hasAttribute('disabled')) {
      $attrs['disabled'] = 'disabled';
    }
    return '<select '.  implode(' ', $attrs).'>'.$options.'</select>'. "\n";
  }
  
  /**
   * 
   * @param array $pattern
   * @param \IZON\Forms\Fields\BaseField $field
   * @param array $params
   * @return string
   */
  protected static function parseRadio($pattern, Fields\BaseField $field, $params = array()) {
    $attrs = self::getAttrs($pattern['attributes'], $field, $params['attributes']);
    $labelPos   = $pattern['textPosition'];
    if(empty($labelPos)) {
      $labelPos = self::LABEL_TEXT_AFTER_FIELD;
    }
    $separator  = !empty($params['separator']) ? $params['separator'] : $pattern['separator'];
    if(empty($separator)) {
      $separator = ' ';
    }
    foreach($field->getOptions() as $key => $value) {
        $swap = '<input '.
                      ((string)$key == (string)$field->getValue() ? ' checked="checked" ' : '') .
                      ($key == $field->getRequired() ? ' required ' : '') .
                      'value="'.$key.'" '.
                      'name="'.$field->getFormName().'" '.
                      'type="radio">';
      $caption = '<span>'.$value.'</span>';
      if($labelPos == self::LABEL_TEXT_AFTER_FIELD) {
        $swap .= $caption;
      } else {
        $swap = $caption.$swap;
      }
      $labelClass = 'form__item--radio';
      if($pattern['label']['class']) {
        $labelClass = $pattern['label']['class'];
      }
      $ret[] = '<label class="'.$labelClass.'">'. $swap .' </label>';
    }
    return implode($separator, $ret);
  }
  
  /**
   * 
   * @param array $pattern
   * @param Fields\CheckboxListField $field
   * @param array $params
   * @return string
   */
  protected static function parseCheckboxListField($pattern, Fields\CheckboxListField $field, $params = array()) {
    $attrs = self::getAttrs($pattern['attributes'], $field, $params['attributes']);
    $labelPos   = $pattern['textPosition'];
    if( empty($labelPos) ) {
      $labelPos = self::LABEL_TEXT_AFTER_FIELD;
    }
    $separator  = !empty($params['separator']) ? $params['separator'] : $pattern['separator'];
    if(empty($separator)) {
      $separator = ' ';
    }
    
    $ret = [];
    foreach($field->getOptions() as $key => $value) {
        $swap = '<input '.
                      ($value["checked"] ? 'checked="checked"' : '').
                      'name="'. $field->getFormName() ."[". $key .']" '.
                      'value="'. $key .'" '.
                      'type="checkbox" >';
      $caption = '<span>'. $value['label'] .'</span>';
      if($labelPos == self::LABEL_TEXT_AFTER_FIELD) {
        $swap .= $caption;
      } else {
        $swap = $caption . $swap;
      }
      $labelClass = $pattern['label']['class'].(isset($params['label']['class']) ? ' '.$params['label']['class'] : '' );
      $ret[] = '<label class="'. $labelClass .'">'. $swap .' </label>';
    }
    return implode($separator, $ret);
  }
  
  
  
  
  /**
  * ocisti/nahradi string o skarede znaky
  * 
  * @param string $string
  */
  public static function clearString($string) {
    return str_replace(array('"'), array('&quot;'), $string);
  }
  /**
   * 
   * @param array $attrs atributy z definice formulare
   * @param Fields\BaseField $field
   * @param array $params parametry/atributy, ktere byly predany fci getField
   * @return string
   */
  protected static function getAttrs($_attrs, $field, $params = []) {
    $str  = [];
    $attrs = self::mergeAttributes(self::mergeAttributes($_attrs, $field->getAttributes()), $params);
    //$attrs = self::mergeAttributes($attrs, $params);
    if($field->hasErrors()) {
      $attrs['class'] .= ' error';
    }
    if($field->isRequired()) {
      $attrs['required'] = 'required';
    }
    $attrs['name'] = $field->getFormName();
    
    if(!empty($attrs) && is_array($attrs)) {
      foreach ($attrs as $key => $value) {
        $str[] = $key.'="'.$value.'"';
      }
    }
    return $str;
  }
  
  protected static function mergeAttributes($patternAttrs, $paramAttrs = []) {
    if(!empty($paramAttrs) && is_array($paramAttrs)) {
      $css = $patternAttrs['class'].(!empty($paramAttrs['class']) ? (empty($patternAttrs['class']) ? '' : ' ').$paramAttrs['class'] : '');
      $patternAttrs = array_merge((array)$patternAttrs, (array)$paramAttrs);
      $patternAttrs['class'] = $css;
    }
    return $patternAttrs;
  }
  /**
   * 
   * @param type $fieldType
   * @param type $type
   * @param type $templateFile
   * @param type $params
   * @todo Dosud nedoimplementovano - nepouzivat
   */
  public static function renderField($fieldType, $type, $templateFile, $params) {
    $template = self::loadTemplate($type, $templateFile);
    $label = self::parseLabel(get_object_vars($template->$fieldType->label), $params['label']);
    $input = self::parseLabel(get_object_vars($template->$fieldType->label), $params['label']);
  }
}
