<?php

namespace IZON\Filesystem\Files;

use Exception;
use IZON\Filesystem\FilesystemInterface;
use IZON\IO\FileHelperTrait;
use IZON\IO\FileInterface;
use League\Flysystem\Util;


class File implements FileInterface {

    /**
     * @var FilesystemInterface
     */
    protected $filesystem;

    /**
     * @var string
     */
    protected $path;

    /**
     * File constructor.
     * @param FilesystemInterface $filesystem
     * @param string $path
     */
    public function __construct(FilesystemInterface $filesystem, string $path) {
        $this->filesystem = $filesystem;
        $this->path = $path;
    }

    use FileHelperTrait;


    public function getFileName() {
        $pathInfo = $this->getPathInfo();
        return implode('.', array_filter([$pathInfo['filename'], $pathInfo['extension']]));
    }

    public function getFileExtension() {
        $pathInfo = $this->getPathInfo();
        return $pathInfo['extension'];
    }

    public function getFileSize() {
        return $this->filesystem->getSize($this->path);
    }

    public function getMimeType() {
        return $this->filesystem->getMimetype($this->path);
    }

    public function getFileContents(): string {
        $content = $this->filesystem->read($this->path);
        if($content === false) {
            // NOTE: throw more specific exception
            throw new Exception("Reading of file '{$this->path}' faild.");
        }
        return $content;
    }

    public function getFileStream() {
        $resource = $this->filesystem->readStream($this->path);
        if($resource === false) {
            // NOTE: throw more specific exception
            throw new Exception("Reading of file '{$this->path}' faild.");
        }
        return $resource;
    }

    public function getTimestamp(): int {
        return $this->filesystem->getTimestamp($this->path);
    }

    public function getOriginUID(): ?string {
        return $this->filesystem->getFilesystemUID();
    }

    public function getFileUID(): string {
        return $this->path;
    }

    protected function getPathInfo() {
        return Util::pathinfo($this->path);
    }

    protected function getMetadata() {
        return $this->filesystem->getMetadata($this->path);
    }

}
