<?php


namespace IZON\Filesystem;

use IZON\Filesystem\Exceptions\UndefinedFilesystemException;


class FilesystemSource implements FilesystemSourceInterface {

    /**
     * @var FilesystemInterface
     */
    protected $defaultFilesystem;
    /**
     * @var FilesystemInterface[]
     */
    protected $filesystems = [];

    /**
     * FilesystemSource constructor.
     * @param FilesystemInterface $defaultFilesystem
     */
    public function __construct(FilesystemInterface $defaultFilesystem) {
        $this->defaultFilesystem = $defaultFilesystem;
    }

    /**
     * @inheritDoc
     */
    public function hasFilesystem(?string $uid = null): bool {
        return empty($uid) || array_key_exists($uid, $this->filesystems);
    }

    /**
     * @return FilesystemInterface
     */
    public function getDefaultFilesystem(): FilesystemInterface {
        return $this->defaultFilesystem;
    }

    /**
     * @param string|null $uid
     * @return FilesystemInterface
     * @throws UndefinedFilesystemException
     */
    public function getFilesystem(?string $uid = null): FilesystemInterface {
        if(empty($uid)) {
            return $this->defaultFilesystem;
        }
        if(!$this->hasFilesystem($uid)) {
            throw new UndefinedFilesystemException("Filesystem with uid '$uid' not registered");
        }
        return $this->filesystems[$uid];
    }

    /**
     * @param FilesystemInterface $filesystem
     */
    public function registerDefaultFilesystem(FilesystemInterface $filesystem): void {
        $this->defaultFilesystem = $filesystem;
    }

    /**
     * @param string $uid
     * @param FilesystemInterface $filesystem
     */
    public function registerFilesystem(string $uid, FilesystemInterface $filesystem): void {
        $this->filesystems[$uid] = $filesystem;
        $filesystem->registerUID($uid);
    }
}
