<?php

declare(strict_types=1);

namespace IZON\DI;

use DI\Definition\EnvironmentVariableDefinition;
use DI\Definition\Helper\AutowireDefinitionHelper;
use DI\Definition\Helper\CreateDefinitionHelper;
use DI\Definition\Reference;
use DI\Definition\StringDefinition;
use DI\Definition\ValueDefinition;
use IZON\DI\Definition\Helper\CreateFactoryObjectDefinitionHelper;
use IZON\DI\Definition\Helper\FactoryDefinitionHelper as IZONFactoryDefinitionHelper;

abstract class DI
{
    /**
     * Helper for defining a value.
     *
     * @param mixed $value
     */
    public static function value($value): ValueDefinition
    {
        return \DI\value($value);
    }

    /**
     * Helper for defining an object.
     *
     * @param string|null $className Class name of the object.
     *                               If null, the name of the entry (in the container) will be used as class name.
     */
    public static function create(?string $className = null): CreateDefinitionHelper
    {
        return \DI\create($className);
    }

    /**
     * Helper for autowiring an object.
     *
     * @param string|null $className Class name of the object.
     *                               If null, the name of the entry (in the container) will be used as class name.
     */
    public static function autowire(?string $className = null): AutowireDefinitionHelper
    {
        return \DI\autowire($className);
    }


    /**
     * Helper for defining a container entry using a factory function/callable.
     *
     * @param callable $factory The factory is a callable that takes the container as parameter
     *                          and returns the value to register in the container.
     */
    public static function factory($factory): IZONFactoryDefinitionHelper
    {
        return new IZONFactoryDefinitionHelper($factory);
    }

    /**
     * Helper for defining factory an object.
     * @param string $className factory class fo create final object
     * @param string|null $factoryMethodName Factory method name to be called on object to create final object
     * @return CreateFactoryObjectDefinitionHelper
     */
    public static function factoryObject(
        string $className,
        ?string $factoryMethodName = null
    ): CreateFactoryObjectDefinitionHelper {
        return new CreateFactoryObjectDefinitionHelper($className, $factoryMethodName);
    }


    /**
     * Helper for referencing another container entry in an object definition.
     */
    public static function get(string $entryName): Reference
    {
        return \DI\get($entryName);
    }


    /**
     * Helper for referencing environment variables.
     *
     * @param string $variableName The name of the environment variable.
     * @param mixed $defaultValue The default value to be used if the environment variable is not defined.
     */
    public static function env(string $variableName, $defaultValue = null): EnvironmentVariableDefinition
    {
        return \DI\env($variableName, $defaultValue);
    }


    /**
     * Helper for concatenating strings.
     *
     * Example:
     *
     *     'log.filename' => DI\string('{app.path}/app.log')
     *
     * @param string $expression A string expression. Use the `{}` placeholders to reference other container entries.
     */
    public static function string(string $expression): StringDefinition
    {
        return \DI\string($expression);
    }
}
