<?php



namespace IZON\DI\Definition\Helper;

use DI\Definition\ObjectDefinition;
use DI\Definition\ObjectDefinition\MethodInjection;
use DI\Definition\ObjectDefinition\PropertyInjection;
use DI\Definition\Source\DefinitionArray;

/**
 * Description of FactoryObjectDefinition
 *
 * @author drymld
 */
class FactoryObjectDefinition extends ObjectDefinition
{
    /**
     * Class name (if null, then the class name is $name).
     * @var class-string|null
     */
    protected ?string $factoryClassName;

    /**
     * @param string $name Entry name
     */
    public function __construct(string $name, ?string $className = null, ?string $factoryClassName = null)
    {
        parent::__construct($name, $className);
        $this->factoryClassName = $factoryClassName;
    }


    public function getClassName(): string
    {
        if ($this->className !== null) {
            return $this->className;
        }

        return $this->getName();
    }

    public function getFactoryClassName(): string
    {
        return $this->factoryClassName;
    }

    /**
     * @return PropertyInjection[] Property injections
     */
    public function getPropertyInjections(): array
    {
        return $this->propertyInjections;
    }

    public function addPropertyInjection(PropertyInjection $propertyInjection): void
    {
        $className = $propertyInjection->getClassName();
        if ($className) {
            // Index with the class name to avoid collisions between parent and
            // child private properties with the same name
            $key = $className . '::' . $propertyInjection->getPropertyName();
        } else {
            $key = $propertyInjection->getPropertyName();
        }

        $this->propertyInjections[$key] = $propertyInjection;
    }

    /**
     * @return MethodInjection[] Method injections
     */
    public function getMethodInjections(): array
    {
        // Return array leafs
        $injections = [];
        array_walk_recursive($this->methodInjections, function ($injection) use (&$injections) {
            $injections[] = $injection;
        });

        return $injections;
    }

    public function replaceNestedDefinitions(callable $replacer): void
    {
        array_walk($this->propertyInjections, function (PropertyInjection $propertyInjection) use ($replacer) {
            $propertyInjection->replaceNestedDefinition($replacer);
        });

        if ($this->constructorInjection) {
            $this->constructorInjection->replaceNestedDefinitions($replacer);
        }

        array_walk($this->methodInjections, function ($injectionArray) use ($replacer) {
            array_walk($injectionArray, function (MethodInjection $methodInjection) use ($replacer) {
                $methodInjection->replaceNestedDefinitions($replacer);
            });
        });
    }

    /**
     * Replaces all the wildcards in the string with the given replacements.
     *
     * @param string[] $replacements
     */
    public function replaceWildcards(array $replacements): void
    {
        $className = $this->getClassName();

        foreach ($replacements as $replacement) {
            $pos = strpos($className, DefinitionArray::WILDCARD);
            if ($pos !== false) {
                $className = substr_replace($className, $replacement, $pos, 1);
            }
        }

        $this->setClassName($className);
    }
}
