<?php

namespace IZON\DI\Definition\Helper;

use DI\Definition\Exception\InvalidDefinition;
use DI\Definition\Helper\CreateDefinitionHelper;
use DI\Definition\ObjectDefinition;
use DI\Definition\ObjectDefinition\MethodInjection;
use DI\Definition\ObjectDefinition\PropertyInjection;
use ReflectionException;
use ReflectionParameter;

/**
 * reimplements \DI\Definition\Helper\FactoryDefinitionHelper to be serializable
 */
class CreateFactoryObjectDefinitionHelper extends CreateDefinitionHelper
{
    public const DEFINITION_CLASS = FactoryObjectDefinition::class;

    /**
     * @var string|null
     */
    private ?string $className;

    /**
     * @var string|null
     */
    protected ?string $factoryClassName;

    /**
     * @var bool|null
     */
    private ?bool $lazy = null;

    /**
     * Array of properties and their value.
     * @var array
     */
    private array $properties = [];

    /**
     * Helper for defining an object.
     *
     * @param string|null $className Class name of the object.
     *                               If null, the name of the entry (in the container) will be used as class name.
     */
    public function __construct(?string $className = null, $factoryClassName = null)
    {
        parent::__construct($className);
        $this->className = $className;
        $this->factoryClassName = $factoryClassName;
    }

    /**
     * @return ObjectDefinition
     */
    public function getDefinition(string $entryName): ObjectDefinition
    {
        $class = $this::DEFINITION_CLASS;
        /** @var ObjectDefinition $definition */
        $definition = new $class($entryName, $this->className, $this->factoryClassName);

        if ($this->lazy !== null) {
            $definition->setLazy($this->lazy);
        }

        if (!empty($this->constructor)) {
            $parameters = $this->fixParameters($definition, '__construct', $this->constructor);
            $constructorInjection = MethodInjection::constructor($parameters);
            $definition->setConstructorInjection($constructorInjection);
        }

        if (!empty($this->properties)) {
            foreach ($this->properties as $property => $value) {
                $definition->addPropertyInjection(
                    new PropertyInjection($property, $value)
                );
            }
        }

        if (!empty($this->methods)) {
            foreach ($this->methods as $method => $calls) {
                foreach ($calls as $parameters) {
                    $parameters = $this->fixParameters($definition, $method, $parameters);
                    $methodInjection = new MethodInjection($method, $parameters);
                    $definition->addMethodInjection($methodInjection);
                }
            }
        }

        return $definition;
    }

    /**
     * Fixes parameters indexed by the parameter name -> reindex by position.
     *
     * This is necessary so that merging definitions between sources is possible.
     *
     * @throws InvalidDefinition
     */
    private function fixParameters(ObjectDefinition $definition, string $method, array $parameters): array
    {
        $fixedParameters = [];

        foreach ($parameters as $index => $parameter) {
            // Parameter indexed by the parameter name, we reindex it with its position
            if (is_string($index)) {
                $callable = [$definition->getClassName(), $method];

                try {
                    $reflectionParameter = new ReflectionParameter($callable, $index);
                } catch (ReflectionException $e) {
                    throw InvalidDefinition::create($definition, sprintf("Parameter with name '%s' could not be found. %s.", $index, $e->getMessage()));
                }

                $index = $reflectionParameter->getPosition();
            }

            $fixedParameters[$index] = $parameter;
        }

        return $fixedParameters;
    }
}
