<?php

namespace IZON\DI\Definition\Helper;

use DI\Definition\Definition;
use DI\Definition\FactoryDefinition;
use DI\Definition\Helper\AutowireDefinitionHelper;
use DI\Definition\Helper\DefinitionHelper;

/**
 * reimplements \DI\Definition\Helper\FactoryDefinitionHelper to be serializable
 */
class CreateFactoryObjectDefinitionHelper implements DefinitionHelper
{
    protected AutowireDefinitionHelper $autowireHelper;

    /**
     * @var string|null
     */
    private ?string $factoryMethodName;

    /**
     * Helper for defining an object.
     *
     * @param string|null $className Class name of the object.
     *                               If null, the name of the entry (in the container) will be used as class name.
     */
    public function __construct(?string $className = null, ?string $factoryMethodName = null)
    {
        $this->autowireHelper = new AutowireDefinitionHelper($className);

        $this->factoryMethodName = $factoryMethodName;
    }

    /**
     * Define the entry as lazy.
     *
     * A lazy entry is created only when it is used, a proxy is injected instead.
     *
     * @return $this
     */
    public function lazy(): self
    {
        $this->autowireHelper->lazy();
        return $this;
    }

    /**
     * Defines the arguments to use to call the constructor.
     *
     * This method takes a variable number of arguments, example:
     *     ->constructor($param1, $param2, $param3)
     *
     * @param mixed... $parameters Parameters to use for calling the constructor of the class.
     *
     * @return $this
     */
    public function constructor(...$parameters): self
    {
        $this->autowireHelper->constructor(...$parameters);
        return $this;
    }

    /**
     * Defines a value to inject in a property of the object.
     *
     * @param string $property Entry in which to inject the value.
     * @param mixed  $value    Value to inject in the property.
     *
     * @return $this
     */
    public function property(string $property, $value): self
    {
        $this->autowireHelper->property($property, $value);
        return $this;
    }

    /**
     * Defines a method to call and the arguments to use.
     *
     * This method takes a variable number of arguments after the method name, example:
     *
     *     ->method('myMethod', $param1, $param2)
     *
     * Can be used multiple times to declare multiple calls.
     *
     * @param string $method       Name of the method to call.
     * @param mixed... $parameters Parameters to use for calling the method.
     *
     * @return $this
     */
    public function method(string $method, ...$parameters): self
    {
        $this->autowireHelper->method($method, ...$parameters);
        return $this;
    }

    public function getDefinition(string $entryName): Definition
    {
        $factoryName = $entryName . 'Factory';
        $factoryObjectDefinition = $this->autowireHelper->getDefinition($factoryName);

        // rovnak na ohejbak pro pripad ze IZON\DI\Definition\Resolver\FactoryObjectCreator
        // zavola \IZON\DI\FactoryObject::getObject
        if (is_subclass_of($factoryObjectDefinition->getClassName(), 'IZON\DI\FactoryObject')) {
            return $factoryObjectDefinition;
        }

        $factoryMethodName = $this->factoryMethodName;
        if ($this->factoryMethodName === null) {
            $reflectionClass = new \ReflectionClass($factoryObjectDefinition->getClassName());
            if ($reflectionClass->hasMethod('create')) {
                $factoryMethodName = 'create';
            } elseif ($reflectionClass->hasMethod('getObject')) {
                $factoryMethodName = 'getObject';
            } elseif ($reflectionClass->hasMethod('__invoke')) {
                $factoryMethodName = '__invoke';
            } else {
                throw new \InvalidArgumentException(
                    "No factory method name provided and no default method"
                    . "(create, getObject, __invoke) found in factory class " . $factoryObjectDefinition->getClassName()
                );
            }
        }

        return new FactoryDefinition(
            $entryName,
            function ($factory, $factoryMethodName) {
                return $factory->{$factoryMethodName}();
            },
            ['factory' => $factoryObjectDefinition, 'factoryMethodName' => $factoryMethodName]
        );
    }

    /**
     * Defines a value for a specific argument of the constructor.
     *
     * This method is usually used together with attributes or autowiring, when a parameter
     * is not (or cannot be) type-hinted. Using this method instead of constructor() allows to
     * avoid defining all the parameters (letting them being resolved using attributes or autowiring)
     * and only define one.
     *
     * @param string|int $parameter Parameter name of position for which the value will be given.
     * @param mixed $value Value to give to this parameter.
     *
     * @return $this
     */
    public function constructorParameter($parameter, $value): self
    {
        $this->autowireHelper->constructorParameter($parameter, $value);
        return $this;
    }

    /**
     * Defines a method to call and a value for a specific argument.
     *
     * This method is usually used together with attributes or autowiring, when a parameter
     * is not (or cannot be) type-hinted. Using this method instead of method() allows to
     * avoid defining all the parameters (letting them being resolved using attributes or
     * autowiring) and only define one.
     *
     * If multiple calls to the method have been configured already (e.g. in a previous definition)
     * then this method only overrides the parameter for the *first* call.
     *
     * @param string $method Name of the method to call.
     * @param string|int $parameter Parameter name of position for which the value will be given.
     * @param mixed $value Value to give to this parameter.
     *
     * @return $this
     */
    public function methodParameter(string $method, $parameter, $value): self
    {
        $this->autowireHelper->methodParameter($method, $parameter, $value);
        return $this;
    }
}
