<?php

namespace IZON\DI\Definition\Resolver;


use Psr\Container\ContainerInterface;

use DI\Definition\Definition;
use DI\Proxy\ProxyFactory;
use DI\Definition\Resolver\DefinitionResolver;
use DI\Definition\Resolver\ArrayResolver;
use DI\Definition\Resolver\FactoryResolver;
use DI\Definition\Resolver\DecoratorResolver;
use DI\Definition\Resolver\InstanceInjector;
use DI\Definition\Resolver\EnvironmentVariableResolver;
use DI\Definition\Resolver\SelfResolver;

/**
 * Container pro pristup k predkum
 * 
 * upravena trida z \DI\Definition\Resolver\DefinitionResolver
 * 
 */
class ResolverDispatcher implements DefinitionResolver {
    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var ProxyFactory
     */
    protected $proxyFactory;

    /**
     * @var SelfResolver
     */
    protected $selfResolvingResolver;
    
    /**
     * @var ArrayResolver
     */
    protected $arrayResolver;
    
    /**
     * @var FactoryResolver
     */
    protected $factoryResolver;
    
    /**
     * @var DecoratorResolver
     */
    protected $decoratorResolver;
    
    /**
     * @var SelfResolver
     */
    protected $objectResolver;
    
    /**
     * @var InstanceInjector
     */
    protected $instanceResolver;
    
    /**
     * @var EnvironmentVariableResolver
     */
    protected $envVariableResolver;
    
    
    public function __construct(ContainerInterface $container, ProxyFactory $proxyFactory) {
        $this->container = $container;
        $this->proxyFactory = $proxyFactory;
    }

    /**
     * Resolve a definition to a value.
     *
     * @param Definition $definition Object that defines how the value should be obtained.
     * @param array      $parameters Optional parameters to use to build the entry.
     *
     * @throws InvalidDefinition If the definition cannot be resolved.
     *
     * @return mixed Value obtained from the definition.
     */
    public function resolve(Definition $definition, array $parameters = []) {
        // Special case, tested early for speed
        if($definition instanceof \DI\Definition\SelfResolvingDefinition) {
            return $definition->resolve($this->container);
        }

        $definitionResolver = $this->getDefinitionResolver($definition);

        return $definitionResolver->resolve($definition, $parameters);
    }

    public function isResolvable(Definition $definition, array $parameters = []): bool {
        // Special case, tested early for speed
        if($definition instanceof \DI\Definition\SelfResolvingDefinition) {
            return $definition->isResolvable($this->container);
        }

        $definitionResolver = $this->getDefinitionResolver($definition);

        return $definitionResolver->isResolvable($definition, $parameters);
    }

    /**
     * Returns a resolver capable of handling the given definition.
     *
     * @throws \RuntimeException No definition resolver was found for this type of definition.
     */
    private function getDefinitionResolver(Definition $definition) : DefinitionResolver
    {
        switch (true) {
            case $definition instanceof \IZON\DI\Definition\Helper\FactoryObjectDefinition:
                if ( !$this->objectResolver ) {
                    $this->objectResolver = new FactoryObjectCreator($this, $this->proxyFactory);
                }

                return $this->objectResolver;
            case $definition instanceof \DI\Definition\ObjectDefinition:
                if ( !$this->objectResolver ) {
                    $this->objectResolver = new FactoryObjectCreator($this, $this->proxyFactory);
                }

                return $this->objectResolver;
            case $definition instanceof \DI\Definition\DecoratorDefinition:
                if (! $this->decoratorResolver) {
                    $this->decoratorResolver = new DecoratorResolver($this->container, $this);
                }

                return $this->decoratorResolver;
            case $definition instanceof \DI\Definition\FactoryDefinition:
                if (! $this->factoryResolver) {
                    $this->factoryResolver = new FactoryResolver($this->container, $this);
                }

                return $this->factoryResolver;
            case $definition instanceof \DI\Definition\ArrayDefinition:
                if (! $this->arrayResolver) {
                    $this->arrayResolver = new ArrayResolver($this);
                }

                return $this->arrayResolver;
            case $definition instanceof \DI\Definition\EnvironmentVariableDefinition:
                if (! $this->envVariableResolver) {
                    $this->envVariableResolver = new EnvironmentVariableResolver($this);
                }

                return $this->envVariableResolver;
            case $definition instanceof \DI\Definition\InstanceDefinition:
                if (! $this->instanceResolver) {
                    $this->instanceResolver = new InstanceInjector($this, $this->proxyFactory);
                }

                return $this->instanceResolver;
            default:
                throw new \RuntimeException('No definition resolver was configured for definition of type ' . get_class($definition));
        }
    }
}

