<?php

namespace IZON\DI\Definition\Resolver;

use DI\Definition\ArrayDefinition;
use DI\Definition\DecoratorDefinition;
use DI\Definition\Definition;
use DI\Definition\EnvironmentVariableDefinition;
use DI\Definition\Exception\DefinitionException;
use DI\Definition\FactoryDefinition;
use DI\Definition\InstanceDefinition;
use DI\Definition\ObjectDefinition;
use DI\Definition\Resolver\ArrayResolver;
use DI\Definition\Resolver\DecoratorResolver;
use DI\Definition\Resolver\DefinitionResolver;
use DI\Definition\Resolver\EnvironmentVariableResolver;
use DI\Definition\Resolver\FactoryResolver;
use DI\Definition\Resolver\InstanceInjector;
use DI\Definition\Resolver\SelfResolver;
use DI\Definition\SelfResolvingDefinition;
use DI\Proxy\ProxyFactory;
use Psr\Container\ContainerInterface;
use RuntimeException;

/**
 * Container pro pristup k predkum
 *
 * upravena trida z \DI\Definition\Resolver\DefinitionResolver
 *
 */
class ResolverDispatcher implements DefinitionResolver
{
    protected ContainerInterface $container;

    /**
     * @var ProxyFactory
     */
    protected $proxyFactory;

    /**
     * @var SelfResolver
     */
    protected $selfResolvingResolver;

    /**
     * @var ArrayResolver
     */
    protected $arrayResolver;

    /**
     * @var FactoryResolver
     */
    protected $factoryResolver;

    /**
     * @var DecoratorResolver
     */
    protected $decoratorResolver;

    /**
     * @var SelfResolver
     */
    protected $objectResolver;

    /**
     * @var InstanceInjector
     */
    protected $instanceResolver;

    /**
     * @var EnvironmentVariableResolver
     */
    protected $envVariableResolver;


    public function __construct(ContainerInterface $container, ProxyFactory $proxyFactory)
    {
        $this->container = $container;
        $this->proxyFactory = $proxyFactory;
    }

    /**
     * Resolve a definition to a value.
     *
     * @param Definition $definition Object that defines how the value should be obtained.
     * @param array $parameters Optional parameters to use to build the entry.
     *
     * @return mixed Value obtained from the definition.
     * @throws DefinitionException If the definition cannot be resolved.
     *
     */
    public function resolve(Definition $definition, array $parameters = [])
    {
        $definitionResolver = $this->getDefinitionResolver($definition);

        return $definitionResolver->resolve($definition, $parameters);
    }

    /**
     * Returns a resolver capable of handling the given definition.
     *
     * @param Definition $definition
     *
     * @return DefinitionResolver
     * @throws RuntimeException No definition resolver was found for this type of definition.
     */
    protected function getDefinitionResolver(Definition $definition)
    {
        switch (true) {
            case $definition instanceof SelfResolvingDefinition:
                if (!$this->selfResolvingResolver) {
                    $this->selfResolvingResolver = new SelfResolver($this->container);
                }

                return $this->selfResolvingResolver;
            case $definition instanceof ObjectDefinition:
                if (!$this->objectResolver) {
                    $this->objectResolver = new FactoryObjectObjectCreator($this, $this->proxyFactory);
                    //                    $this->objectResolver = new ObjectCreator($this, $this->proxyFactory);
                }

                return $this->objectResolver;
            case $definition instanceof DecoratorDefinition:
                if (!$this->decoratorResolver) {
                    $this->decoratorResolver = new DecoratorResolver($this->container, $this);
                }

                return $this->decoratorResolver;
            case $definition instanceof FactoryDefinition:
                if (!$this->factoryResolver) {
                    $this->factoryResolver = new FactoryResolver($this->container, $this);
                }

                return $this->factoryResolver;
            case $definition instanceof ArrayDefinition:
                if (!$this->arrayResolver) {
                    $this->arrayResolver = new ArrayResolver($this);
                }

                return $this->arrayResolver;
            case $definition instanceof EnvironmentVariableDefinition:
                if (!$this->envVariableResolver) {
                    $this->envVariableResolver = new EnvironmentVariableResolver($this);
                }

                return $this->envVariableResolver;
            case $definition instanceof InstanceDefinition:
                if (!$this->instanceResolver) {
                    $this->instanceResolver = new InstanceInjector($this, $this->proxyFactory);
                }

                return $this->instanceResolver;
            default:
                throw new RuntimeException('No definition resolver was configured for definition of type ' . get_class($definition));
        }
    }

    /**
     * Check if a definition can be resolved.
     *
     * @param Definition $definition Object that defines how the value should be obtained.
     * @param array $parameters Optional parameters to use to build the entry.
     *
     * @return bool
     */
    public function isResolvable(Definition $definition, array $parameters = [])
    {
        $definitionResolver = $this->getDefinitionResolver($definition);

        return $definitionResolver->isResolvable($definition, $parameters);
    }
}
