<?php

declare(strict_types=1);

namespace IZON\DI;

use DI\Definition\EntryReference;
use DI\Definition\Helper\EnvironmentVariableDefinitionHelper;
use DI\Definition\Helper\ObjectDefinitionHelper;
use DI\Definition\Helper\StringDefinitionHelper;
use DI\Definition\Helper\ValueDefinitionHelper;
use IZON\DI\Definition\Helper\FactoryDefinitionHelper;

abstract class DI
{
    /**
     * Helper for defining a value.
     *
     * @param mixed $value
     */
    public static function value($value): ValueDefinitionHelper
    {
        return \DI\value($value);
    }

    /**
     * Helper for defining an object.
     *
     * @param string|null $className Class name of the object.
     *                               If null, the name of the entry (in the container) will be used as class name.
     */
    public static function create(?string $className = null): ObjectDefinitionHelper
    {
        return \DI\object($className);
    }

    /**
     * Helper for autowiring an object.
     *
     * @param string|null $className Class name of the object.
     *                               If null, the name of the entry (in the container) will be used as class name.
     */
    public static function autowire(?string $className = null): ObjectDefinitionHelper
    {
        return \DI\object($className);
    }


    /**
     * Helper for defining a container entry using a factory function/callable.
     *
     * @param callable $factory The factory is a callable that takes the container as parameter
     *                          and returns the value to register in the container.
     */
    public static function factory($factory): FactoryDefinitionHelper
    {
        return new FactoryDefinitionHelper($factory);
    }


    /**
     * Helper for referencing another container entry in an object definition.
     */
    public static function get(string $entryName): EntryReference
    {
        return \DI\get($entryName);
    }


    /**
     * Helper for referencing environment variables.
     *
     * @param string $variableName The name of the environment variable.
     * @param mixed $defaultValue The default value to be used if the environment variable is not defined.
     */
    public static function env(string $variableName, $defaultValue = null): EnvironmentVariableDefinitionHelper
    {
        return \DI\env($variableName, $defaultValue);
    }


    /**
     * Helper for concatenating strings.
     *
     * Example:
     *
     *     'log.filename' => DI\string('{app.path}/app.log')
     *
     * @param string $expression A string expression. Use the `{}` placeholders to reference other container entries.
     */
    public static function string(string $expression): StringDefinitionHelper
    {
        return \DI\string($expression);
    }
}
