<?php

namespace IZON\DB\Types;

use DateTime;
use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Types\Type;
use IZON\DB\Exceptions\DBException;
use IZON\Utils\Date;

/**
 * Date type for mapping to db.
 */
class DateType extends Type
{
    public const IZON_DATE = 'izon_date';

    /**
     * {@inheritdoc}
     */
    public function getName(): string
    {
        return self::IZON_DATE;
    }

    /**
     * {@inheritdoc}
     */
    public function getSQLDeclaration(array $fieldDeclaration, AbstractPlatform $platform): string
    {
        return $platform->getDateTypeDeclarationSQL($fieldDeclaration);
    }

    /**
     * {@inheritdoc}
     */
    public function convertToPHPValue($value, AbstractPlatform $platform)
    {
        if ($value === null) {
            return $value;
        }
        $format = $this->getFormat();
        $format = '!' . $format; # Exclamation mark resets time.
        $phpDateTime = DateTime::createFromFormat($format, $value);
        if ($phpDateTime === false) {
            throw new DBException('Unsupported date `' . $value . '` format from DB. Expected format ' . $format);
        }
        $date = new Date($phpDateTime);
        return $date;
    }

    /**
     * Gets date format string.
     * @return string
     */
    protected function getFormat(): string
    {
        return 'Y-m-d';
    }

    /**
     * {@inheritdoc}
     */
    public function convertToDatabaseValue($value, AbstractPlatform $platform)
    {
        if ($value === null) {
            return $value;
        }
        if (!$value instanceof Date) {
            throw new DBException('Unsupported date class ' . get_class($value));
        }
        /** @var Date $value */
        $dateString = $value->format($this->getFormat());
        return $dateString;
    }
}
