<?php

namespace IZON\DB\Repository\Factories;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Proxy\ProxyFactory;
use Doctrine\ORM\Repository\RepositoryFactory;
use Doctrine\Persistence\ObjectRepository;
use ProxyManager\Configuration;
use ProxyManager\Factory\AccessInterceptorValueHolderFactory;
use ProxyManager\FileLocator\FileLocator;
use ProxyManager\GeneratorStrategy\FileWriterGeneratorStrategy;
use ReflectionClass;
use ReflectionException;

class AutobindQueryRepositoryFactory implements RepositoryFactory
{
    protected ?AccessInterceptorValueHolderFactory $proxyFactory = null;

    /**
     * The list of EntityRepository instances.
     *
     * @var ObjectRepository[]
     */
    protected array $repositoryList = [];

    /**
     * {@inheritdoc}
     */
    public function getRepository(EntityManagerInterface $entityManager, $entityName): EntityRepository
    {
        $repositoryHash = $entityManager->getClassMetadata($entityName)->getName() . spl_object_hash($entityManager);

        if (isset($this->repositoryList[$repositoryHash])) {
            return $this->repositoryList[$repositoryHash];
        }

        return $this->repositoryList[$repositoryHash] = $this->createRepository($entityManager, $entityName);
    }

    /**
     * Create a new repository instance for an entity class.
     *
     * @param EntityManagerInterface $entityManager The EntityManager instance.
     * @param string $entityName The name of the entity.
     *
     * @return EntityRepository
     *
     * @throws ReflectionException
     */
    protected function createRepository(EntityManagerInterface $entityManager, string $entityName): EntityRepository
    {
        $metadata = $entityManager->getClassMetadata($entityName);
        $repositoryClassName = $metadata->customRepositoryClassName
            ?: $entityManager->getConfiguration()->getDefaultRepositoryClassName();

        $baseRepository = new $repositoryClassName($entityManager, $metadata);

        $reflection = new ReflectionClass($repositoryClassName);

        $proxy = $this->getProxyFactory($entityManager)->createProxy($baseRepository);

        $postInterceptor = function ($proxy, $instance, $method, $params, $returnValue, &$returnEarly) {
            $instance->bindParametersToQuery($returnValue, $params);
        };

        foreach ($reflection->getMethods() as $method) {
            if (in_array($method->getName(), ['find', 'findAll', 'findBy', 'findOneBy'])) {
                continue;
            }
            if (!str_starts_with($method->getName(), 'find')) {
                continue;
            }
            $proxy->setMethodSuffixInterceptor($method->getName(), $postInterceptor); // @phpstan-ignore argument.type
        }
        return $proxy;
    }

    protected function getProxyFactory(EntityManagerInterface $entityManager): AccessInterceptorValueHolderFactory
    {
        if ($this->proxyFactory === null) {
            $ormConfig = $entityManager->getConfiguration();
            $autoGenerateProxyClasses = $entityManager->getConfiguration()->getAutoGenerateProxyClasses();

            $proxyDir = null;
            if ($ormConfig instanceof \IZON\DB\EntityManager\Configuration) {
                $proxyDir = $ormConfig->getRepositoryProxiesDir();
            }

            $config = new Configuration();

            if (
                $proxyDir !== null
                && $autoGenerateProxyClasses == ProxyFactory::AUTOGENERATE_FILE_NOT_EXISTS
            ) { // generate proxy files only if configured for production file generateion
                // create directory if not exists
                if (!file_exists($proxyDir)) {
                    mkdir($proxyDir, 0775, true);
                }
                // generate the proxies and store them as files
                $fileLocator = new FileLocator($proxyDir);
                $config->setGeneratorStrategy(new FileWriterGeneratorStrategy($fileLocator));
                $config->setProxiesNamespace($ormConfig->getProxyNamespace());
                // set the directory to read the generated proxies from
                $config->setProxiesTargetDir($proxyDir);
                // then register the autoloader
                spl_autoload_register($config->getProxyAutoloader()); // @phpstan-ignore argument.type
            }

            $this->proxyFactory = new AccessInterceptorValueHolderFactory($config);
        }
        return $this->proxyFactory;
    }
}
