<?php

namespace IZON\DB\Repository;

use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use IZON\DB\Paginator\PageContent;
use IZON\DB\Paginator\PageContentInterface;
use IZON\DB\Paginator\PaginatorConfig;
use IZON\DB\Paginator\PaginatorConfigInterface;
use IZON\DB\QueryFactory\DefaultQueryFactory;
use IZON\DB\QueryFactory\QueryFactoryInterface;
use IZON\DB\QueryParams\QueryParamsInterface;
use IZON\Utils\ArrayUtils;
use IZON\Utils\Date;
use Psr\Log\LoggerInterface;

/**
 * @template T of object
 * @extends EntityRepository<T>
 * @implements RepositoryInterface<T>
 */
class BaseRepository extends EntityRepository implements RepositoryInterface
{
    protected LoggerInterface $log;

    protected QueryFactoryInterface $defaultPaginatorQueryFactory;

    /**
     *
     * @param EntityManagerInterface $em
     * @param ClassMetadata $classMetadata
     */
    public function __construct(EntityManagerInterface $em, ClassMetadata $classMetadata)
    {
        parent::__construct($em, $classMetadata);
        $this->defaultPaginatorQueryFactory = new DefaultQueryFactory($classMetadata);
    }

    /**
     * Creates default PaginatorConfig
     *
     * @param int $pageSize
     * @return PaginatorConfigInterface
     */
    public function createDefaultPaginatorConfig(int $pageSize): PaginatorConfigInterface
    {
        $queryFactory = $this->defaultPaginatorQueryFactory;

        $paginatorConfig = new PaginatorConfig(
            'findDefaultPaginatorQuery',
            $queryFactory->getParameterDefinitionNames(),
            $queryFactory->getControlParameterDefinitionNames(),
            $queryFactory->getOrderDefinitions()
        );

        $paginatorConfig->setPageSize($pageSize);

        return $paginatorConfig;
    }

    /**
     *
     * @param PaginatorConfigInterface $paginatorConfig
     * @return PageContentInterface
     */
    public function paginate(PaginatorConfigInterface $paginatorConfig): PageContentInterface
    {
        $queryParams = $paginatorConfig->toQueryParams();

        $query = $this->{$paginatorConfig->getQueryName()}($queryParams);
        $this->bindParametersToQuery($query, [$queryParams]);

        # Set LIMIT, OFFSET
        if ($query instanceof Query) {
            $query->setFirstResult($paginatorConfig->getFirstResult());
            $query->setMaxResults($paginatorConfig->getPageSize());
        }

        # For getting total count
        $doctrinePaginator = new Paginator($query);

        $pageContent = new PageContent(count($doctrinePaginator), $paginatorConfig->getPageSize(), $query->getResult());

        return $pageContent;
    }

    /**
     * binds parameters to query
     * @param AbstractQuery $query
     * @param array<int, mixed> $params
     */
    public function bindParametersToQuery(AbstractQuery $query, array $params)
    {
        if (
            count($params) === 1
            && ArrayUtils::first($params) instanceof QueryParamsInterface
        ) {
            $queryParams = ArrayUtils::first($params);
            foreach ($queryParams->getParameterNames() as $paramName) {
                $query->setParameter($paramName, $queryParams->getParameter($paramName));
            }
        } else {
            foreach ($params as $name => $value) {
                if (is_null($value)) {
                    continue;
                }
                if ($value instanceof Date) {
                    $value = $value->getDateTime();
                }
                $query->setParameter($name, $value);
            }
        }
    }

    protected function findDefaultPaginatorQuery(QueryParamsInterface $queryParams): AbstractQuery
    {
        return $this->defaultPaginatorQueryFactory->getQuery($this->getEntityManager(), $queryParams);
    }
}
