<?php

namespace IZON\DB\QueryParams;

use IZON\DB\Exceptions\DBException;

/**
 * QueryParams
 * Encapsulates parameters passed to query factory methods for creating query.
 * @author Aleš Kopecký <kopecky@izon.cz>
 */
class QueryParams implements QueryParamsInterface
{
    /**
     * @var array<string, mixed> query parameters passed to db
     */
    protected array $parameters = [];

    /**
     * @var array<string, mixed> control parameters to use for construction of query, not passed to db
     */
    protected array $controlParameters = [];

    /**
     * @var OrderBy[]
     */
    protected array $orders = [];

    /**
     *
     * @param array<string, mixed> $params
     * @param array<string, mixed> $controlParams
     * @param OrderBy[] $orders
     */
    public function __construct(array $params = [], array $controlParams = [], array $orders = [])
    {
        $this->parameters = $params;
        $this->controlParameters = $controlParams;
        $this->orders = $orders;
    }


    /**
     * @return mixed[]
     */
    public function getParameters(): array
    {
        return $this->parameters;
    }

    /**
     * @param string $paramName
     * @return bool
     */
    public function hasParameter(string $paramName): bool
    {
        return array_key_exists($paramName, $this->parameters);
    }

    /**
     * @param string $prefix
     * @return bool
     */
    public function hasParameterWithPrefix(string $prefix): bool
    {
        return !empty($this->getParameterNamesWithPrefix($prefix));
    }

    /**
     * @param string $prefix
     * @return array
     */
    public function getParameterNamesWithPrefix(string $prefix): array
    {
        return array_filter(
            $this->getParameterNames(),
            function ($name) use ($prefix) {
                return str_starts_with($name, $prefix);
            }
        );
    }

    /**
     * @return string[]
     */
    public function getParameterNames(): array
    {
        return array_keys($this->parameters);
    }

    /**
     * @param string $paramName
     * @return mixed
     * @throws DBException
     */
    public function getParameter(string $paramName)
    {
        if (!array_key_exists($paramName, $this->parameters)) {
            throw new DBException('Parameter ' . $paramName . ' not set');
        }
        return $this->parameters[$paramName];
    }

    /**
     *
     * @param string $paramName
     * @param mixed $value
     */
    public function putParameter(string $paramName, $value)
    {
        $this->parameters[$paramName] = $value;
    }

    /**
     * @return mixed[]
     */
    public function getControlParameters(): array
    {
        return $this->controlParameters;
    }

    /**
     * @return string[]
     */
    public function getControlParameterNames(): array
    {
        return array_keys($this->parameters);
    }


    /**
     * @param string $paramName
     * @return bool
     */
    public function hasControlParameter(string $paramName): bool
    {
        return array_key_exists($paramName, $this->controlParameters);
    }

    /**
     * @param string $paramName
     * @return mixed
     * @throws DBException
     */
    public function getControlParameter(string $paramName)
    {
        if (!array_key_exists($paramName, $this->controlParameters)) {
            throw new DBException('Parameter ' . $paramName . ' not set');
        }
        return $this->controlParameters[$paramName];
    }

    /**
     * @param string $paramName
     * @param mixed $value
     */
    public function putControlParameter(string $paramName, $value)
    {
        $this->controlParameters[$paramName] = $value;
    }

    /**
     * @return OrderBy[]
     */
    public function getOrders(): array
    {
        return $this->orders;
    }

    /**
     * @param string $orderName
     * @return bool
     */
    public function hasOrder(string $orderName): bool
    {
        return array_key_exists($orderName, $this->orders);
    }

    /**
     * @param string $orderName
     * @param string $orientation
     */
    public function addOrder(string $orderName, string $orientation)
    {
        $this->addOrderBy(new OrderBy($orderName, $orientation));
    }

    /**
     * @param OrderBy $orderBy
     */
    public function addOrderBy(OrderBy $orderBy)
    {
        $this->orders[$orderBy->getName()] = $orderBy;
    }
}
