<?php

namespace IZON\DB\QueryFactoryReader;

use Doctrine\ORM\Mapping\ClassMetadata;
use IZON\DB\Exceptions\DBException;
use IZON\DB\QueryFactory\DefaultQueryFactory;
use IZON\DB\QueryFactory\QueryFactory;
use IZON\DB\QueryFactory\QueryFactoryInterface;

/**
 * QueryFactoryReaderInterface
 * Converts queryFactory file to object of QueryFactoryInterface.
 *
 * @author Aleš Kopecký <kopecky@izon.cz>
 */
class RepositoryQueryFactoryReader implements QueryFactoryConfigReaderInterface
{
    /**
     * @var DefaultQueryFactory[]
     */
    protected array $defaultClassQueryFactoriesCache = [];

    /**
     * @var array[]
     */
    protected array $classQueryFactoriesCache = [];

    //
    //    public function getDefaultQueryFactory(string $className, ClassMetadata $classMetadata): QueryFactoryInterface {
    //
    //    }

    public function getQueryFactory(ClassMetadata $classMetadata, string $queryName): QueryFactoryInterface
    {
        $className = $classMetadata->getName();

        if ($queryName === QueryFactoryInterface::DEFAULT_QUERY_NAME) {
            if (!array_key_exists($className, $this->defaultClassQueryFactoriesCache)) {
                $this->defaultClassQueryFactoriesCache[$className] = $this->createDefaultQueryFactory($classMetadata);
            }
            return $this->defaultClassQueryFactoriesCache[$className];
        } else {
            if (!array_key_exists($className, $this->classQueryFactoriesCache)) {
                $this->classQueryFactoriesCache[$className] = $this->createQueryFactories($classMetadata);
            }

            if (array_key_exists($queryName, $this->classQueryFactoriesCache[$className])) {
                return $this->classQueryFactoriesCache[$className][$queryName];
            }

            throw new DBException('There is not query factory for query ' . $queryName . ' degifined for class ' . $classMetadata->getName());
        }
    }

    protected function createDefaultQueryFactory(ClassMetadata $classMetadata): DefaultQueryFactory
    {
        $defaultQueryFactory = new DefaultQueryFactory($classMetadata);
        if ($classMetadata->customRepositoryClassName !== null
            && method_exists($classMetadata->customRepositoryClassName, 'modifyDefaultQueryFactory')) {
            // DOTO: test if method is static
            $classMetadata->customRepositoryClassName::modifyDefaultQueryFactory($defaultQueryFactory, $classMetadata);
        }

        return $defaultQueryFactory;
    }

    /**
     *
     * @param ClassMetadata $classMetadata
     * @return array<string, QueryFactory>
     */
    protected function createQueryFactories(ClassMetadata $classMetadata): array
    {
        $queryFactoriesHash = [];

        if (
            $classMetadata->customRepositoryClassName !== null
            && method_exists($classMetadata->customRepositoryClassName, 'modifyDefaultQueryFactory')
        ) {
            // DOTO: test if method is static
            /** @var QueryFactory[] $queryFactories */
            $queryFactories = $classMetadata->customRepositoryClassName::loadQueryFactories($classMetadata);
            foreach ($queryFactories as $queryFactory) {
                $queryFactoriesHash[$queryFactory->getQueryName()] = $queryFactory;
            }
        }

        return $queryFactoriesHash;
    }
}
