<?php

namespace IZON\DB\QueryFactory;

use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\ClassMetadata;
use IZON\DB\Exceptions\DBException;
use IZON\DB\Paginator\PaginatorConfigInterface;
use IZON\DB\QueryParams\OrderBy;
use IZON\DB\QueryParams\QueryParamsInterface;

class DefaultQueryFactory implements QueryFactoryInterface
{
    public const DEFAULT_VALUE_PREFIX = '%';

    public const DEFAULT_VALUE_SUFFIX = '%';

    public const DEFAULT_EMPTY_CAN_BE_NULL = false;

    /**
     * @var ClassMetadata
     */
    protected $classMetadata;

    /**
     * @var array[]
     */
    protected array $parameterDefinitions = [];

    /**
     * @var array<int, array{name: string, excluded: bool}>
     */
    protected array $orderDefinitions;

    protected bool $isInitialized = false;

    /**
     * @param ClassMetadata $classMetadata
     */
    public function __construct(ClassMetadata $classMetadata)
    {
        $this->classMetadata = $classMetadata;
    }

    public function getQueryName(): string
    {
        return PaginatorConfigInterface::DEFAULT_QUERY_NAME;
    }

    /**
     * {@intheritDoc}
     * @param EntityManagerInterface $em
     * @param QueryParamsInterface $queryParams
     * @return AbstractQuery
     * @throws DBException
     */
    public function getQuery(EntityManagerInterface $em, QueryParamsInterface $queryParams): AbstractQuery
    {
        $this->init();

        $qb = $em->createQueryBuilder();

        $alias = 'alias';
        $qb->select('alias')
            ->from($this->classMetadata->getName(), $alias);

        $whereArray = [];
        foreach ($queryParams->getParameters() as $paramName => $value) {
            if (!array_key_exists($paramName, $this->parameterDefinitions)) {
                # Param does not exist in entity class
                throw new DBException("Parameter {$paramName} does not exist in entity class {$this->classMetadata->getName()}.");
            }

            # Gathering data for WHERE part
            $columnName = $alias . '.' . $paramName;
            $parameterDefinition = $this->parameterDefinitions[$paramName];

            # Composing WHERE part
            $wherePart = "(";
            if ($parameterDefinition['comparator'] == "LIKE") { // is something that can be compared as string
                $likePrefix = $parameterDefinition['likePrefix'];
                $likeSuffix = $parameterDefinition['likeSuffix'];
                $wherePart .= $columnName . " LIKE CONCAT('" . $likePrefix . "', :" . $paramName . ", '" . $likeSuffix . "')";
            } else {
                $wherePart .= $columnName . " " . $parameterDefinition['comparator'] . " :" . $paramName;
            }

            $emptyCanBeNull = $parameterDefinition['emptyCanBeNull'];
            if ($emptyCanBeNull) {
                $wherePart .= " OR (" . ":" . $paramName . "='' AND " . $columnName . " IS NULL)";
            }
            $wherePart .= ")";

            $whereArray[] = $wherePart;
        }
        if (!empty($whereArray)) {
            $qb->where(implode(' AND ', $whereArray));
        }

        # Orders
        /** @var OrderBy $order */
        foreach ($queryParams->getOrders() as $order) {
            $qb->addOrderBy($alias . '.' . $order->getName(), $order->getType());
        }

        return $qb->getQuery();
    }

    protected function init()
    {
        if ($this->isInitialized) {
            return;
        }

        foreach ($this->classMetadata->getFieldNames() as $fieldName) {
            $fieldMapping = $this->classMetadata->getFieldMapping($fieldName);

            $parameterDefinition = [
                'name' => $fieldName,
                'comparator' => '=',
                'excluded' => false,
                'likePrefix' => '',
                'likeSuffix' => '',
                'emptyCanBeNull' => self::DEFAULT_EMPTY_CAN_BE_NULL,
            ];
            if (in_array($fieldMapping['type'], [Types::STRING])) { // is something that can be compared as string
                $parameterDefinition['comparator'] = "LIKE";
                $parameterDefinition['likePrefix'] = self::DEFAULT_VALUE_PREFIX;
                $parameterDefinition['likeSuffix'] = self::DEFAULT_VALUE_SUFFIX;
            }
            $this->parameterDefinitions[$fieldName] = $parameterDefinition;
            $this->orderDefinitions[$fieldName] = [
                'name' => $fieldName,
                'excluded' => false,
            ];
        }

        foreach ($this->classMetadata->getAssociationMappings() as $associationMapping) {
            if ($associationMapping['type'] == ClassMetadata::MANY_TO_ONE) {
                $fieldName = $associationMapping['fieldName'];
                $this->parameterDefinitions[$fieldName] = [
                    'name' => $fieldName,
                    'comparator' => "=",
                    'excluded' => false,
                    'likePrefix' => self::DEFAULT_VALUE_PREFIX,
                    'likeSuffix' => self::DEFAULT_VALUE_SUFFIX,
                    'emptyCanBeNull' => self::DEFAULT_EMPTY_CAN_BE_NULL,
                ];
            }
        }

        $this->isInitialized = true;
    }

    /**
     * excludes property from parameters
     * @param string $propertyName
     */
    public function excludePropertyParameter(string $propertyName): void
    {
        $this->init();

        // DOTO: test $propertyName existence
        $this->parameterDefinitions[$propertyName]['excluded'] = true;
    }

    /**
     * excludes property from orders
     * @param string $propertyName
     */
    public function excludePropertyOrder(string $propertyName): void
    {
        $this->init();

        // DOTO: test $propertyName existence
        $this->orderDefinitions[$propertyName]['excluded'] = true;
    }

    /**
     *
     * @param string $propertyName
     * @return string
     */
    public function getParameterPrefix(string $propertyName): string
    {
        $this->init();

        // DOTO: test $propertyName existence
        return $this->parameterDefinitions[$propertyName]['likePrefix'];
    }

    /**
     *
     * @param string $propertyName
     * @param string $prefix
     */
    public function setParameterPrefix(string $propertyName, string $prefix): void
    {
        $this->init();

        // DOTO: test $propertyName existence
        $this->parameterDefinitions[$propertyName]['likePrefix'] = $prefix;
    }

    /**
     *
     * @param string $propertyName
     * @return string
     */
    public function getParameterSuffix(string $propertyName): string
    {
        $this->init();

        // DOTO: test $propertyName existence
        return $this->parameterDefinitions[$propertyName]['likeSuffix'];
    }

    /**
     *
     * @param string $propertyName
     * @param string $suffix
     */
    public function setParameterSuffix(string $propertyName, string $suffix): void
    {
        $this->init();

        // DOTO: test $propertyName existence
        $this->parameterDefinitions[$propertyName]['likeSuffix'] = $suffix;
    }

    public function setParameterEmptyCanBeNull(string $propertyName, bool $active): void
    {
        $this->init();

        // DOTO: test $propertyName existence
        $this->parameterDefinitions[$propertyName]['emptyCanBeNull'] = $active;
    }


    /**
     * {@inheritDoc}
     */
    public function getParameterDefinitionNames(): array
    {
        $this->init();

        $fitered = array_filter($this->parameterDefinitions, function ($definition) {
            return !$definition['excluded'];
        });
        return array_map(function ($definition) {
            return $definition['name'];
        }, $fitered);
    }

    /**
     * {@inheritDoc}
     */
    public function getControlParameterDefinitionNames(): array
    {
        return [];
    }

    /**
     * {@inheritDoc}
     */
    public function getOrderDefinitions(): array
    {
        $this->init();

        $filtered = array_filter($this->orderDefinitions, function ($definition) {
            return !$definition['excluded'];
        });
        return array_map(function ($definition) {
            return $definition['name'];
        }, $filtered);
    }
}
