<?php

namespace IZON\DB\Paginator;

use IZON\DB\Exceptions\DBException;
use IZON\DB\QueryParams\OrderBy;
use IZON\DB\QueryParams\QueryParams;

/**
 * slouzi jako konfigurace strankovani
 *
 * TODO: mozna vyresit inteligentneji nastavovani ostatnich parametru
 */
class PaginatorConfig implements PaginatorConfigInterface
{
    protected string $queryName;

    /**
     * @var string[] regexes defining parametes
     */
    protected array $parameterDefinitions = [];

    /**
     * @var string[] regexes defining control parameters to use for construction of query, not passed to db
     */
    protected array $controlParameterDefinitions = [];

    /**
     * @var string[] regexes defining orders that can be used
     */
    protected array $orderDefinitions = [];

    /**
     * @var array<string, mixed>
     */
    protected array $parameters = [];

    /**
     * @var array<string, mixed>
     */
    protected $controlParameters = [];

    /**
     * @var OrderBy[]
     */
    protected array $orders = [];

    protected int $pageSize = PaginatorConfigInterface::DEFAULT_PAGE_SIZE;

    protected int $firstResult = 0;


    public function __construct(
        $queryName,
        $parameterDefinitions,
        $controlParameterDefinitions,
        $orderDefinitions
    ) {
        $this->queryName = $queryName;
        $this->parameterDefinitions = $parameterDefinitions;
        $this->controlParameterDefinitions = $controlParameterDefinitions;
        $this->orderDefinitions = $orderDefinitions;
    }

    public function getPageSize(): int
    {
        return $this->pageSize;
    }

    public function setPageSize(int $pageSize)
    {
        $this->pageSize = $pageSize;
    }

    public function getQueryName(): string
    {
        return $this->queryName;
    }

    /**
     * @param string $propertyName
     * @return bool
     */
    public function hasParameter(string $propertyName): bool
    {
        return array_key_exists($propertyName, $this->parameters)
            || array_key_exists($propertyName, $this->controlParameters);
    }

    /**
     * @param string $paramName
     * @throws DBException
     */
    public function removeParameter(string $paramName)
    {
        if (!$this->__isset($paramName)) {
            throw new DBException("Parameter $paramName not defined");
        }
        if ($this->isParameter($paramName)) {
            unset($this->parameters[$paramName]);
        } else {
            unset($this->controlParameters[$paramName]);
        }
    }

    /**
     *
     * @param string $propertyName
     * @return boolean
     * @throws DBException
     */
    public function __isset(string $propertyName): bool
    {
        return $this->isParameter($propertyName)
            || $this->isControlParameter($propertyName);
    }

    protected function isParameter(string $paramName): bool
    {
        foreach ($this->parameterDefinitions as $definedFieldName) {
            $result = preg_match("#^" . $definedFieldName . "$#u", $paramName);
            if ($result === false) {
                throw new DBException("Can't match deffined field " . $definedFieldName . " pattern");
            } elseif ($result) {
                return true;
            }
        }
        return false;
    }

    protected function isControlParameter(string $paramName): bool
    {
        foreach ($this->controlParameterDefinitions as $definedFieldName) {
            $result = preg_match("#^" . $definedFieldName . "$#u", $paramName);
            if ($result === false) {
                throw new DBException("Can't match defined field " . $definedFieldName . " pattern");
            } elseif ($result) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return array pole se vsemi nastavenymi parametry
     */
    public function getParameters(): array
    {
        $parameters = [];
        foreach ($this->parameters as $propertyName => $value) {
            if ($this->__isset($propertyName)) {
                $parameters[$propertyName] = $value;
            }
        }
        foreach ($this->controlParameters as $propertyName => $value) {
            if ($this->__isset($propertyName)) {
                $parameters[$propertyName] = $value;
            }
        }
        return $parameters;
    }

    /**
     * To enable calling get*() and set*().
     * For setting properties using getters and setters.
     *
     * @param string $methodName
     * @param array $arguments
     * @throws DBException If property is not defined.
     */
    public function __call(string $methodName, array $arguments)
    {
        if (
            !str_starts_with($methodName, 'get')
            && !str_starts_with($methodName, 'set')
        ) {
            throw new DBException('method ' . $methodName . " is not supported. Only getters and setters are supported");
        }

        $propertyNamePart = substr($methodName, 3); # Removes first three characters ('get' or 'set')
        $propertyName = lcfirst($propertyNamePart); # Lower case first letter

        # Getter
        if (str_starts_with($methodName, 'get')) {
            # If field is defined
            if ($this->__isset($propertyName)) {
                return $this->getParameter($propertyName);
            } else {
                throw new DBException("Property $propertyName not defined");
            }
        }

        # Setter
        if (str_starts_with($methodName, 'set')) {
            # If field is defined
            if ($this->__isset($propertyName)) {
                $value = $arguments[0]; # Get value
                $this->putParameter($propertyName, $value);
            } else {
                throw new DBException("Property $propertyName not defined");
            }
        }
    }

    /**
     * @param string $paramName
     * @return mixed
     * @throws DBException
     */
    public function getParameter(string $paramName)
    {
        if (!$this->__isset($paramName)) {
            throw new DBException("Parameter $paramName not defined");
        }
        if ($this->isParameter($paramName)) {
            return $this->parameters[$paramName] ?? null;
        } else {
            return $this->controlParameters[$paramName] ?? null;
        }
    }

    /**
     * @param string $paramName
     * @param mixed $paramValue
     * @throws DBException
     */
    public function putParameter(string $paramName, $paramValue)
    {
        if (!$this->__isset($paramName)) {
            throw new DBException("Parameter $paramName not defined");
        }
        if ($this->isParameter($paramName)) {
            $this->parameters[$paramName] = $paramValue;
        } else {
            $this->controlParameters[$paramName] = $paramValue;
        }
    }

    /**
     * @return OrderBy[]
     */
    public function getOrders(): array
    {
        return $this->orders;
    }

    /**
     * Sets ordering by given property and given direction.
     *
     * @param string $orderName
     * @param string $direction OrderBy::ORDER_ASC|OrderBy::ORDER_SESC
     * @throws DBException
     */
    public function addOrder(string $orderName, string $direction)
    {
        if (!$this->isOrder($orderName)) {
            throw new DBException("Property '{$orderName}' is not in defined orders and therefore cannot be used for orders.");
        }
        $orderBy = new OrderBy($orderName, $direction);
        $this->orders[] = $orderBy;
    }

    /**
     * @param string $orderName
     * @return bool
     */
    public function isOrder(string $orderName): bool
    {
        foreach ($this->orderDefinitions as $orderDefinition) {
            $result = preg_match("#^" . $orderDefinition . "$#u", $orderName);
            if ($result === false) {
                throw new DBException("Can't match to deffined order " . $orderDefinition . " pattern");
            } elseif ($result) {
                return true;
            }
        }
        return false;
    }

    public function clearOrders()
    {
        $this->orders = [];
    }

    /**
     * @return int
     */
    public function getFirstResult(): int
    {
        return $this->firstResult;
    }

    /**
     * @param int $firstResult
     */
    public function setFirstResult(int $firstResult)
    {
        $this->firstResult = $firstResult;
    }

    /**
     * {@inheritedDoc}
     */
    public function toQueryParams(): QueryParams
    {
        $queryParams = new QueryParams($this->parameters, $this->controlParameters, $this->orders);
        return $queryParams;
    }
}
