<?php

namespace IZON\DB\Naming\Impl;

use Doctrine\ORM\Mapping\NamingStrategy;
use Doctrine\ORM\Mapping\UnderscoreNamingStrategy;

/**
 * default naming strategy we use to map object to tables
 */
class DefaultNamingStrategy extends UnderscoreNamingStrategy implements NamingStrategy
{
    /**
     * @var CASE_LOWER|CASE_UPPER
     */
    private int $case;

    /**
     * Underscore naming strategy construct.
     *
     * @param CASE_LOWER|CASE_UPPER $case
     */
    public function __construct($case = CASE_LOWER)
    {
        parent::__construct($case);
        $this->case = $case;
    }

    /**
     * @return CASE_LOWER|CASE_UPPER
     */
    public function getCase(): int
    {
        return $this->case;
    }

    /**
     * Sets string case CASE_LOWER | CASE_UPPER.
     * Alphabetic characters converted to lowercase or uppercase.
     *
     * @param CASE_LOWER|CASE_UPPER $case
     *
     * @return void
     */
    public function setCase(int $case): void
    {
        $this->case = $case;
    }

    /**
     * {@inheritdoc}
     */
    public function propertyToColumnName(string $propertyName, string $className): string
    {
        return $this->underscore($propertyName);
    }

    /**
     * @param string $string
     *
     * @return string
     */
    private function underscore(string $string): string
    {
        $string = preg_replace('/(?<=[a-z])([A-Z])/', '_$1', $string);

        if ($this->case === CASE_UPPER) {
            return strtoupper($string);
        }

        return strtolower($string);
    }

    /**
     * {@inheritdoc}
     */
    public function embeddedFieldToColumnName(
        string $propertyName,
        string $embeddedColumnName,
        string $className,
        string $embeddedClassName,
    ): string {
        return $this->underscore($propertyName) . '_' . $embeddedColumnName;
    }

    //    /**
    //     * {@inheritdoc}
    //     */
    //    public function referenceColumnName(): string
    //    {
    //        return $this->case === CASE_UPPER ? 'ID' : 'id';
    //    }

    /**
     * {@inheritdoc}
     */
    public function joinColumnName(string $propertyName, string $className): string
    {
        return 'fk_' . $this->underscore($propertyName) . '_' . $this->referenceColumnName();
    }

    /**
     * {@inheritdoc}
     */
    public function joinTableName(
        string $sourceEntity,
        string $targetEntity,
        string $propertyName,
    ): string {
        return $this->classToTableName($sourceEntity) . '_' . $this->classToTableName($targetEntity);
    }

    /**
     * {@inheritdoc}
     */
    public function classToTableName(string $className): string
    {
        if (strpos($className, '\\') !== false) {
            $className = substr($className, strrpos($className, '\\') + 1);
        }

        return $this->underscore($className) . 's';
    }

    /**
     * {@inheritdoc}
     */
    public function joinKeyColumnName(
        string $entityName,
        string|null $referencedColumnName,
    ): string {
        if (strpos($entityName, '\\') !== false) {
            $entityName = substr($entityName, strrpos($entityName, '\\') + 1);
        }
        $entityColumnName = $this->underscore($entityName);

        return 'fk_' . $entityColumnName . '_' .
            ($referencedColumnName ?: $this->referenceColumnName());
    }
}
