<?php

namespace IZON\DB\Connections\Factories;

use Doctrine\Common\EventSubscriber;
use Doctrine\DBAL\DriverManager;
use Doctrine\DBAL\Logging\SQLLogger;
use Doctrine\DBAL\Types\Type;
use IZON\DB\Behaviour\BehaviourInterface;
use IZON\DB\ConnectionInterface;
use IZON\DB\Connections\Configuration;
use IZON\DB\Connections\Connection;
use IZON\DB\Exceptions\DBException;
use IZON\DB\Logger\PsrSQLLogger;
use IZON\DB\Types\DateTimeType;
use IZON\DB\Types\DateType;
use IZON\DB\Types\TypeRegistrars\SimpleTypeRegistrar;
use IZON\DB\Types\TypeRegistrars\TypeRegistrarInterface;
use Psr\Log\LoggerInterface;

/**
 * Description of ConnectionFactory
 *
 * @author Lukáš Linhart
 */
class ConnectionFactory
{
    /**
     * @var array<string, mixed>
     */
    protected array $connectionParams;

    /**
     * @var string
     */
    protected string $charset = 'utf8';

    /**
     * @var string|null where to store proxies
     */
    protected ?string $proxyDir = null;

    protected ?SQLLogger $sqlLogger = null;

    /**
     * @var EventSubscriber[]
     */
    protected array $eventSubscribers = [];

    /**
     * @var TypeRegistrarInterface[]
     */
    protected array $typeRegistrars = [];

    /**
     *
     * @param string|array<string, mixed> $host hostname (domain, ip) or array of connection parameters
     * @param null|string $database database name
     * @param null|string $user username
     * @param null|string $password user password
     * @param string $driver default 'pdo_mysql' other options (pdo_sqlite, pdo_pgsql, pdo_oci, oci8, ibm_db2, pdo_sqlsrv, mysqli, drizzle_pdo_mysql, sqlanywhere, sqlsrv)
     */
    public function __construct(
        $host,
        ?string $database = null,
        ?string $user = null,
        ?string $password = null,
        string $driver = 'pdo_mysql'
    ) {
        if (is_array($host)) {
            $this->connectionParams = $host;
        } else {
            $this->connectionParams = [
                'dbname' => $database,
                'user' => $user,
                'password' => $password,
                'host' => $host,
                'driver' => $driver
            ];
        }

        $this->typeRegistrars = [
            new SimpleTypeRegistrar(DateType::IZON_DATE, DateType::class),
            new SimpleTypeRegistrar(DateTimeType::IZON_DATETIME, DateTimeType::class),
        ];
    }

    /**
     *
     * @return ConnectionInterface
     */
    public function create(): ConnectionInterface
    {
        $params = array_merge(
            $this->connectionParams,
            [
                'charset' => $this->charset,
                'wrapperClass' => Connection::class
            ]
        );

        $configuration = new Configuration();
        // configure proxy dir
        $configuration->setProxyDir($this->proxyDir);
        // configure sql logger if exists
        if ($this->sqlLogger instanceof SQLLogger) {
            $configuration->setSQLLogger($this->sqlLogger); // @phpstan-ignore method.deprecated
        }
        $connection = DriverManager::getConnection($params, $configuration);
        if (!$connection instanceof ConnectionInterface) {
            throw new DBException('Unsupported connection class ' . get_class($connection));
        }

        // register types
        foreach ($this->typeRegistrars as $typeRegistrar) {
            if (!Type::hasType($typeRegistrar->getTypeName())) {
                Type::addType($typeRegistrar->getTypeName(), $typeRegistrar->getTypeClass());
            }
            $typeRegistrar->initType();
        }

        foreach ($this->eventSubscribers as $subscriber) {
            $connection->getEventManager() // @phpstan-ignore method.deprecated
            ->addEventSubscriber($subscriber);
        }

        $connection
            ->getDatabasePlatform()
            ->registerDoctrineTypeMapping('enum', 'string');

        return $connection;
    }

    /**
     *
     * @param string|null $proxyDir where to store proxies
     */
    public function setProxyDir(?string $proxyDir): void
    {
        $this->proxyDir = $proxyDir;
    }

    /**
     * @param SQLLogger|LoggerInterface $sqlLogger
     */
    public function setSQLLogger($sqlLogger): void // @phpstan-ignore parameter.deprecatedInterface
    {
        if ($sqlLogger instanceof LoggerInterface) {
            $sqlLogger = new PsrSQLLogger($sqlLogger);
        }
        $this->sqlLogger = $sqlLogger;
    }

    /**
     * @param EventSubscriber $subscriber
     */
    public function addEventSubscriber(EventSubscriber $subscriber): void
    {
        $this->eventSubscribers[] = $subscriber;
    }

    /**
     * @param string $charset charset of connection default is utf8
     */
    public function setCharset(string $charset): void
    {
        $this->charset = $charset;
    }

    public function addBehaviour(BehaviourInterface $behaviour): void
    {
        if (!empty($behaviour->getEventSubscriber())) {
            $this->addEventSubscriber($behaviour->getEventSubscriber());
        }
    }

    /**
     * adds type registrar to register additional types
     * @param TypeRegistrarInterface $typeRegistrator
     */
    public function addTypeRegistrar(TypeRegistrarInterface $typeRegistrator): void
    {
        $this->typeRegistrars[] = $typeRegistrator;
    }

    /**
     * @param string $typeName
     * @param string $typeClass
     */
    public function registerType(string $typeName, string $typeClass): void
    {
        $this->typeRegistrars[] = new SimpleTypeRegistrar($typeName, $typeClass);
    }
}
