<?php

namespace IZON\DB\Behaviour\SoftDelete;

use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\PersistentCollection;
use IZON\Utils\ArrayUtils;

trait SoftDeletableTrait // @phpstan-ignore trait.unused
{
    /**
     * @var bool if true, entity will not be deleted from DB
     */
    protected bool $softDelete = true;

    #[ORM\Column]
    protected bool $deleted = false;

    /**
     * filters all soft deleted objects from iterable
     * @param iterable $collection
     * @return array|Collection
     */
    public function filterSoftDeleted(iterable $collection)
    {
        if ($collection instanceof PersistentCollection) {
            $criteria = Criteria::create()->where(Criteria::expr()->eq('deleted', false));
            return $collection->matching($criteria);
        } elseif ($collection instanceof Collection) {
            $collection->filter(function (SoftDeletable $element) {
                return !$element->isDeleted();
            });
        }
        return array_filter(ArrayUtils::toArray($collection), function (SoftDeletable $element) {
            return !$element->isDeleted();
        });
    }

    /**
     * @return bool
     */
    public function isDeleted(): bool
    {
        return $this->deleted ?? false;
    }

    /**
     * @param bool $deleted
     */
    public function setDeleted(bool $deleted): void
    {
        $this->deleted = $deleted;
    }

    public function isSoftDelete(): bool
    {
        return $this->softDelete;
    }

    public function setSoftDelete(bool $softDelete): void
    {
        $this->softDelete = $softDelete;
    }
}
