<?php

namespace IZON\DB\QueryFactory;

use Doctrine\ORM\AbstractQuery;
use IZON\DB\EntityManagerInterface;
use IZON\DB\Exceptions\DBException;
use IZON\DB\QueryParams\QueryParamsInterface;


class QueryFactory implements QueryFactoryInterface {
    
    /**
     * @var string
     */
    protected $queryName;
    
    /**
     * @var callable
     */
    protected $queryFactoryCallable;
    
    /**
     * @var array[]
     */
    protected $parameterDefinitions = [];
    
    /**
     * @var array
     */
    protected $controlParameterDefinitions = [];

    /**
     * @var array[]
     */
    protected $orderDefinitions;

    /**
     * @param array $config
     */
    public function __construct(string $queryName, callable $queryFactoryCallable) {
        $this->queryName = $queryName;
        $this->queryFactoryCallable = $queryFactoryCallable;
    }
    
    public function getQueryName(): string {
        return $this->queryName;
    }
    
    /**
     * Gets query from queryFactory.
     * 
     * @param string $queryName
     * @param EntityManagerInterface $em
     * @param QueryParamsInterface $queryParams
     * @return AbstractQuery
     * @throws DBException
     */
    public function getQuery(EntityManagerInterface $em, QueryParamsInterface $queryParams): AbstractQuery {
        return call_user_func($this->queryFactoryCallable, $em, $queryParams);
    }
    
    /**
     * @param string $paramName
     * @param array $definition
     * @throws DBException
     */
    public function addParameterDefinition(string $paramName, $definition = []) {
        if(isset($this->parameterDefinitions[$paramName])) {
            throw new DBException("Parameter definition for '{$paramName}' is already set.");
        }
        $this->parameterDefinitions[$paramName] = $definition;
    }
    
    /**
     * @param string $paramName
     * @param array $definition
     * @throws DBException
     */
    public function addControlParameterDefinition(string $paramName, $definition = []) {
        if(isset($this->controlParameterDefinitions[$paramName])) {
            throw new DBException("Control parameter definition for '{$paramName}' is already set.");
        }
        $this->controlParameterDefinitions[$paramName] = $definition;
    }
    
    /**
     * @param string $orderName
     * @throws DBException
     */
    public function addOrderDefinition(string $orderName) {
        if(isset($this->orderDefinitions[$orderName])) {
            throw new DBException("Order definition for '{$orderName}' is already set.");
        }
        $this->orderDefinitions[$orderName] = $orderName;
    }
    
    /**
     * {@inheritDoc}
     * @return array
     */
    public function getParameterDefinitionNames(): array {
        return array_keys($this->parameterDefinitions);
    }

    /**
     * {@inheritDoc}
     * @return array
     */
    public function getControlParameterDefinitionNames(): array {
        return array_keys($this->controlParameterDefinitions);
    }

    /**
     * {@inheritDoc}
     * @return array
     */
    public function getOrderDefinitions(): array {
        return array_keys($this->orderDefinitions);
    }
}
