<?php

namespace IZON\DB\Mapping\Driver;

use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping\ClassMetadataInfo;
use Doctrine\ORM\Mapping\Column;
use Doctrine\ORM\Mapping\Driver\AttributeDriver as DoctrineAttributeDriver;
use Doctrine\Persistence\Mapping\ClassMetadata;

/**
 * Attribute driver that expands wildcards $paths
 * and sets repository
 */
class AttributeDriver extends DoctrineAttributeDriver {

    /**
     * @var string suffix to class name to form repository class name
     */
    protected $repositoryClassSuffix = 'Repository';

    /**
     * @var string last part of namespace to form repository name space
     */
    protected $repositoryClassNamespace = "Repositories";

    /**
     * Initializes a new AttributeDriver
     * @param string|string[]|null $paths  One or multiple paths where mapping classes can be found.
     */
    public function __construct($paths = null) {
        $realPaths = [];
        // expand paths with wildcards to real paths
        foreach ($paths as $path) {
            if(!is_dir($path) && str_contains($path, '**')) {
                foreach( glob( str_replace('**', '*', $path) ) as $name) {
                    $realPaths[] = $name;
                }
            } else {
                $realPaths[] = $path;
            }
        }
        parent::__construct($realPaths);
    }

    /**
     * {@inheritDoc}
     */
    public function loadMetadataForClass($className, ClassMetadata $metadata): void {
        parent::loadMetadataForClass($className, $metadata);

        // traverse fields and infer data type if no data type where specified
        $fieldNames = $metadata->getFieldNames();
        foreach($fieldNames as $fieldName) {
            $fieldMapping = $metadata->getFieldMapping($fieldName);

            // set nullable
            $reflectionProperty = $metadata->getReflectionClass()->getProperty($fieldName);
            $reflectionType = $reflectionProperty->getType();
            if( $reflectionType !== null && $reflectionType->allowsNull() ) {
                $fieldMapping['nullable'] = true;
                $metadata->addInheritedFieldMapping($fieldMapping);
            }

            // switch identifier field from int to bigint
            if(
                $this->isSimpleIdentifierField($metadata, $fieldName)
                && $fieldMapping['type'] == Types::INTEGER
            ) {
                // remap id field to bigint if has interger type
                $fieldMapping['type'] = Types::BIGINT;
                $metadata->addInheritedFieldMapping($fieldMapping);
            }
        }

        // set custom repository if exists and not specified explicitly
        if( $metadata->customRepositoryClassName === null ) {
            $repositoryClassName = $this->inferCustomRepositoryClassName($metadata);
            if( $repositoryClassName !== null ) { //
                $metadata->setCustomRepositoryClass($repositoryClassName);
            }
        }
    }

    /**
     * @param ClassMetadata $classMetadata
     * @param string $propertyName
     * @return bool
     */
    protected function isSimpleIdentifierField(ClassMetadata $classMetadata, string $propertyName) {
        // remap id field to bigint if has interger type
        $identifierFieldNames = $classMetadata->getIdentifierFieldNames();
        return count($identifierFieldNames) == 1 && in_array($propertyName, $identifierFieldNames);
    }

    /**
     * @param ClassMetadata $metadata
     * @return string|null
     */
    protected function inferCustomRepositoryClassName(ClassMetadata $metadata) {
        $reflectionClass = $metadata->getReflectionClass();

        $classShortName = $reflectionClass->getShortName();
        $repositoryShortClassName = $classShortName . $this->repositoryClassSuffix;

        $classNamespace = $reflectionClass->getNamespaceName();
        $classNamespaceParts = explode('\\', $classNamespace);
        $repositoryNamespaceParts = array_slice($classNamespaceParts, 0, count($classNamespaceParts)-1); // remove last part
        $repositoryNamespaceParts[] = $this->repositoryClassNamespace;

        $repositoryClassName = implode('\\', $repositoryNamespaceParts) .'\\'. $repositoryShortClassName;

        if( class_exists($repositoryClassName) ) {
            return $repositoryClassName;
        }

        return null;
    }
}
