<?php

namespace IZON\DB\Mapping\Driver;

use Doctrine\Persistence\Mapping\ClassMetadata;
use Doctrine\ORM\Mapping\Driver\AnnotationDriver as DoctrineAnnotationDriver;

/**
 * anotation driver that inferes data types from @var
 */
class AnnotationDriver extends DoctrineAnnotationDriver {

    /**
     * @var string suffix to class name to form repository class name
     */
    protected $repositoryClassSuffix = 'Repository';

    /**
     * @var string last part of namespace to form repository name space
     */
    protected $repositoryClassNamespace = "Repositories";

    /**
     * Initializes a new AnnotationDriver that uses the given AnnotationReader for reading
     * docblock annotations.
     *
     * @param Reader               $reader The AnnotationReader to use, duck-typed.
     * @param string|string[]|null $paths  One or multiple paths where mapping classes can be found.
     */
    public function __construct($reader, $paths = null) {
        parent::__construct($reader, $paths);
        $this->parser = new \PhpDocReader\PhpDocReader();
    }

    /**
     * {@inheritDoc}
     */
    public function loadMetadataForClass($className, ClassMetadata $metadata) {
        parent::loadMetadataForClass($className, $metadata);

        // traverse fields and infer data type if no data type where specified
        $fieldNames = $metadata->getFieldNames();
        foreach($fieldNames as $fieldName) {
            $fieldMapping = $metadata->getFieldMapping($fieldName);
            $propertyReflection = $metadata->getReflectionClass()->getProperty($fieldName);

            $ref = new \IZON\DB\Reflection\ReflectionClass($metadata->getReflectionClass());
            $type = $ref->getPropertyColumnTagType($fieldName);

            $varDataType = $this->getCommentDataType($propertyReflection);
            if( $varDataType !== null // we have var annotation
                && $type == null  // no type defined in Column annotation
                && $fieldMapping['type'] === 'string') { // field is mapped as string
                // remap
                $fieldMapping['name'] = $fieldName;
                $fieldMapping['type'] = $varDataType;

                if( $this->isIdentifierField($metadata, $fieldName)
                    && $varDataType == \Doctrine\DBAL\Types\Type::INTEGER ) {
                    // remap id field to bigint if has interger type
                    $fieldMapping['type'] = \Doctrine\DBAL\Types\Type::BIGINT;
                }

                $metadata->addInheritedFieldMapping($fieldMapping);
            }
        }

        // set custom repository if exists and not specified explicitly
        if( $metadata->customRepositoryClassName === null ) {
            $repositoryClassName = $this->inferCustomRepositoryClassName($metadata);
            if( $repositoryClassName !== null ) { //
                $metadata->setCustomRepositoryClass($repositoryClassName);
            }
        }
    }

    /**
     * @param $metadata
     * @param $propertyName
     * @return bool
     */
    protected function isIdentifierField($metadata, $propertyName) {
        // remap id field to bigint if has interger type
        $identifierFieldNames = $metadata->getIdentifierFieldNames();
        return count($identifierFieldNames) && in_array($propertyName, $identifierFieldNames);
    }

    /**
     * @param ClassMetadata $metadata
     * @return string|null
     */
    protected function inferCustomRepositoryClassName(ClassMetadata $metadata) {
        $reflectionClass = $metadata->getReflectionClass();

        $classShortName = $reflectionClass->getShortName();
        $repositoryShortClassName = $classShortName . $this->repositoryClassSuffix;

        $classNamespace = $reflectionClass->getNamespaceName();
        $classNamespaceParts = explode('\\', $classNamespace);
        $repositoryNamespaceParts = array_slice($classNamespaceParts, 0, count($classNamespaceParts)-1); // remove last part
        $repositoryNamespaceParts[] = $this->repositoryClassNamespace;

        $repositoryClassName = implode('\\', $repositoryNamespaceParts) .'\\'. $repositoryShortClassName;

        if( class_exists($repositoryClassName) ) {
            return $repositoryClassName;
        }

        return null;
    }

    /**
     * skalarni typy promenych v php
     * http://php.net/manual/en/language.types.intro.php
     * @var array
     */
    protected static $scalarTypeMappings = [
        'bool' => 'boolean',
        'boolean' => 'boolean',
        'string' => 'string',
        'int' => 'integer',
        'integer' => 'integer',
        'float' => 'float',
        'double' => 'float',
    ];

    /**
     * vytahne datovy typ z komentare u property v mapovane tride
     * @param ReflectionProperty $property
     * @return string
     */
    protected function getCommentDataType(\ReflectionProperty $property) {
        $reader = self::getDocReader();
        //find existing class
        $dataType = $reader->getPropertyClass($property); // nacte tridu

        //existing class not found -> try to find scalar type
        if($dataType == NULL) {
            if( preg_match('#@var\s+([^\s]+)#', $property->getDocComment(), $matches) ) { // nacte skalarni typ
                list(, $dataType) = $matches;

                $compositeDataTypes = explode('|', $dataType);

                $dataType = null;
                foreach($compositeDataTypes as $compositeDataType) {
                    if( array_key_exists($compositeDataType, self::$scalarTypeMappings) ) {  // use first recognised scalar data type
                        $dataType = self::$scalarTypeMappings[$compositeDataType];
                        break;
                    }
                }
            } else {
                $dataType = NULL;
            }
        }

        return $dataType;
    }

    /**
     * nacita datovy typ z dokumentacniho bloku
     * @var PhpDocReader
     */
    private static $docReader;

    /**
     * @return PhpDocReader
     */
    protected function getDocReader() {
        //TODO: set new phpDocReader
        if(self::$docReader == NULL) {
            self::$docReader = new \PhpDocReader\PhpDocReader();
        }
        return self::$docReader;
    }
}
