<?php
namespace IZON\DB\Repository\Factories;

use Doctrine\Common\Proxy\AbstractProxyFactory;
use ReflectionClass;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\ORM\Repository\RepositoryFactory;
use ProxyManager\Configuration;
use ProxyManager\Factory\AccessInterceptorValueHolderFactory;
use ProxyManager\FileLocator\FileLocator;
use ProxyManager\GeneratorStrategy\FileWriterGeneratorStrategy;

class AutobindQueryRepositoryFactory implements RepositoryFactory {

    /**
     * @var AccessInterceptorValueHolderFactory
     */
    protected $proxyFactory;

    /**
     * The list of EntityRepository instances.
     *
     * @var \Doctrine\Common\Persistence\ObjectRepository[]
     */
    protected $repositoryList = [];

    /**
     * {@inheritdoc}
     */
    public function getRepository(EntityManagerInterface $entityManager, $entityName)
    {
        $repositoryHash = $entityManager->getClassMetadata($entityName)->getName() . spl_object_hash($entityManager);

        if (isset($this->repositoryList[$repositoryHash])) {
            return $this->repositoryList[$repositoryHash];
        }

        return $this->repositoryList[$repositoryHash] = $this->createRepository($entityManager, $entityName);
    }

    /**
     * Create a new repository instance for an entity class.
     *
     * @param \Doctrine\ORM\EntityManagerInterface $entityManager The EntityManager instance.
     * @param string $entityName The name of the entity.
     *
     * @return \Doctrine\Common\Persistence\ObjectRepository
     *
     * @throws ReflectionException
     */
    protected function createRepository(EntityManagerInterface $entityManager, $entityName) {
        /** @var ClassMetadata $metadata  */
        $metadata = $entityManager->getClassMetadata($entityName);
        $repositoryClassName = $metadata->customRepositoryClassName
            ?: $entityManager->getConfiguration()->getDefaultRepositoryClassName();

        $baseRepository = new $repositoryClassName($entityManager, $metadata);

        $reflection = new ReflectionClass($repositoryClassName);

        $proxy = $this->getProxyFactory($entityManager)->createProxy($baseRepository);

        $postInterceptor = function($proxy, $instance, $method, $params, $returnValue, &$returnEarly) {
            $instance->bindParametersToQuery($returnValue, $params);
        };

        foreach($reflection->getMethods() as $method) {
            if(in_array($method->getName(), ['find', 'findAll', 'findBy', 'findOneBy'])) {
                continue;
            }
            if (!\IZON\String\startsWith($method->getName(), 'find')) {
               continue;
            }
            $proxy->setMethodSuffixInterceptor($method->getName(), $postInterceptor);
        }
        return $proxy;
    }

    protected function getProxyFactory(EntityManagerInterface $entityManager): AccessInterceptorValueHolderFactory {
        if( $this->proxyFactory === null ) {
            $ormConfig = $entityManager->getConfiguration();
            $autoGenerateProxyClasses = $entityManager->getConfiguration()->getAutoGenerateProxyClasses();

            $proxyDir = null;
            if( $ormConfig instanceof \IZON\DB\EntityManager\Configuration ) {
                $proxyDir = $ormConfig->getRepositoryProxiesDir();
            }

            $config = new Configuration();

            if( $proxyDir !== null
                && $autoGenerateProxyClasses == AbstractProxyFactory::AUTOGENERATE_FILE_NOT_EXISTS ) { // generate proxy files only if configured for production file generateion
                // create directory if not exists
                if( !file_exists($proxyDir) ) {
                    mkdir($proxyDir, 0775, true);
                }
                // generate the proxies and store them as files
                $fileLocator = new FileLocator($proxyDir);
                $config->setGeneratorStrategy(new FileWriterGeneratorStrategy($fileLocator));
                $config->setProxiesNamespace($ormConfig->getProxyNamespace());
                // set the directory to read the generated proxies from
                $config->setProxiesTargetDir($proxyDir);
                // then register the autoloader
                spl_autoload_register($config->getProxyAutoloader());
            }

            $this->proxyFactory = new AccessInterceptorValueHolderFactory($config);
        }
        return $this->proxyFactory;
    }
}
