<?php

namespace IZON\DB\QueryFactoryReader;

use Doctrine\ORM\Mapping\ClassMetadata;
use IZON\DB\Exceptions\DBException;
use IZON\DB\Paginator\PaginatorConfigInterface;
use IZON\DB\QueryFactory\DefaultQueryFactory;
use IZON\DB\QueryFactory\QueryFactory;
use IZON\DB\QueryFactory\QueryFactoryInterface;
use ReflectionClass;

/**
 * QueryFactoryReaderInterface
 * Converts queryFactory file to object of QueryFactoryInterface.
 *
 * @author Aleš Kopecký <kopecky@izon.cz>
 */
class QueryFactoryReader implements QueryFactoryConfigReaderInterface {

    /**
     * Path to query factory files.
     * Relative to Entity location.
     *
     * @var string
     */
    protected $filesPath = '../DB/';

    /**
     * @var string
     */
    protected $fileNameSuffix = '';

    /**
     * Cached QueryFactory.
     *
     * @var QueryFactoryInterface
     */
    protected $queryFactoryCache;

    /**
     * Cached DefaultQueryFactory.
     *
     * @var QueryFactoryInterface
     */
    protected $defaultQueryFactoryCache;

    public function __construct($filesPath) {
        $this->filesPath = $filesPath;
    }

    /**
     * Tries to find QueryFactory file and converts that into object of QueryFactoryInterface.
     *
     * @param string $className
     * @return QueryFactoryInterface
     * @throws DBException
     */
    public function getQueryFactory(ClassMetadata $classMetadata, string $queryName): QueryFactoryInterface {
        # Is already inited in cache?
        if($this->queryFactoryCache instanceof QueryFactoryInterface) {
            return $this->queryFactoryCache;
        }
        $className = $classMetadata->getName();
        $conf = $this->getConfigArray($className);
        $queryFactory = new QueryFactory($classMetadata, $conf);

        $this->queryFactory = $queryFactory;
        return $this->queryFactory;
    }

    /**
     * Creates default query factory object.
     *
     * @param string $className
     * @param ClassMetadata $classMetadata
     * @return QueryFactoryInterface
     */
    public function getDefaultQueryFactory(string $className, ClassMetadata $classMetadata): QueryFactoryInterface {
        # Is already inited in cache?
        if($this->defaultQueryFactoryCache instanceof DefaultQueryFactory) {
            return $this->defaultQueryFactoryCache;
        }
        try{
            $conf = $this->getConfigArray($className);
        } catch (DBException $ex) {
            # Query factory file was not found
            $conf = [];
        }
        if(array_key_exists(PaginatorConfigInterface::DEFAULT_QUERY_NAME, $conf) ) {
            $conf = $conf[PaginatorConfigInterface::DEFAULT_QUERY_NAME];
        }
        $queryFactory = new DefaultQueryFactory($classMetadata, $conf);

        $this->defaultQueryFactoryCache = $queryFactory;
        return $queryFactory;
    }

    /**
     * Returns configuration array from file if file found.
     *
     * @param string $className
     * @return array
     * @throws DBException If query factory file was not found.
     */
    protected function getConfigArray(string $className): array {
        # Find QF array file for this entity
        $rc = new ReflectionClass($className);
        $dir = dirname($rc->getFileName());

        $filesDirPath = $dir . '/' . $this->filesPath;
        $fileNames = array_diff(scandir($filesDirPath), array('..', '.'));
        foreach($fileNames as $fileName) {
            if($fileName == $rc->getShortName() . $this->fileNameSuffix .'.php') {
                $qfFile = $fileName;
                break;
            }
        }
        if(!isset($qfFile)) {
            //doto - maybe throw somehting like QueryFactoryNotFoundException() ?
            throw new DBException("Query Factory for {$className} was not found in {$filesDirPath}.");
        }
        # Get array from file into variable
        $conf = include $filesDirPath .'/'. $qfFile;
        return $conf;
    }
}
