<?php

namespace IZON\DB\Connections\Factories;

use Doctrine\Common\EventSubscriber;
use Doctrine\DBAL\DriverManager;
use Doctrine\DBAL\Logging\SQLLogger;
use Doctrine\DBAL\Types\Type;
use IZON\DB\Behaviour\BehaviourInterface;
use IZON\DB\ConnectionInterface;
use IZON\DB\Connections\Configuration;
use IZON\DB\Connections\Connection;
use IZON\DB\Exceptions\DBException;
use IZON\DB\Extensions\TypeExtensionInterface;
use IZON\DB\Files\Types\FileType;
use IZON\DB\Files\Types\RasterImageType;
use IZON\DB\Types\DateTimeType;
use IZON\DB\Types\DateType;
use IZON\DB\Types\TypeRegistrars\SimpleTypeRegistrar;
use IZON\DB\Types\TypeRegistrars\TypeRegistrarInterface;

/**
 * Description of ConnectionFactory
 *
 * @author Lukáš Linhart
 */
class ConnectionFactory {

    /**
     * @var string
     */
    protected $host;

    /**
     * @var string
     */
    protected $database;

    /**
     * @var string
     */
    protected $user;

    /**
     * @var string
     */
    protected $password;

    /**
     * @var string
     */
    protected $driver;

    /**
     * @var string
     */
    protected $charset = 'utf8';

    /**
     * @var string|null $proxyDir where to store proxies
     */
    protected $proxyDir = null;

    /**
     * @var SQLLogger
     */
    protected $sqlLogger = null;

    /**
     * @var EventSubscriber
     */
    protected $eventSubscribers = [];

    /**
     * @var TypeRegistrarInterface[]
     */
    protected $typeRegistrars = [];

    /**
     *
     * @param string $host hostname (domain, ip)
     * @param string $database database name
     * @param string $user username
     * @param string $password user password
     * @param string $driver default 'pdo_mysql' other options (pdo_sqlite, pdo_pgsql, pdo_oci, oci8, ibm_db2, pdo_sqlsrv, mysqli, drizzle_pdo_mysql, sqlanywhere, sqlsrv)
     */
    public function __construct(
        string $host,
        string $database,
        string $user,
        string $password,
        string $driver = 'pdo_mysql'
    ) {
        $this->host = $host;
        $this->database = $database;
        $this->user = $user;
        $this->password = $password;
        $this->driver = $driver;

        $this->typeRegistrars = [
            new SimpleTypeRegistrar(DateType::IZON_DATE, DateType::class),
            new SimpleTypeRegistrar(DateTimeType::IZON_DATETIME, DateTimeType::class),
        ];
    }

    /**
     *
     * @return ConnectionInterface
     */
    public function create() : ConnectionInterface {
        $params = [
            'dbname' => $this->database,
            'user' => $this->user,
            'password' => $this->password,
            'host' =>  $this->host,
            'driver' => $this->driver,
            'charset'  => $this->charset,
            'wrapperClass' => Connection::class
        ];

        $configuration = new Configuration();
        // configure proxy dir
        $configuration->setProxyDir($this->proxyDir);
        // configure sql logger if exists
        if($this->sqlLogger instanceof SQLLogger) {
            $configuration->setSQLLogger($this->sqlLogger);
        }
        $connection = DriverManager::getConnection($params, $configuration);
        if( ! $connection instanceof ConnectionInterface ) {
            throw new DBException('Unsupported connection class '. get_class($connection));
        }

        // register types
        foreach( $this->typeRegistrars as $typeRegistrar ) {
            if( !Type::hasType($typeRegistrar->getTypeName()) ) {
                Type::addType($typeRegistrar->getTypeName(), $typeRegistrar->getTypeClass());
            }
            $typeRegistrar->initType();
        }

        // NOTE: kept only for reference, can be removed later
        // DOTO: create something to register types
//        if( !\Doctrine\DBAL\Types\Type::hasType(\IZON\DB\Files\Types\FileType::FILE) ) { // fix multiple app config creation
//            \Doctrine\DBAL\Types\Type::addType(\IZON\DB\Files\Types\FileType::FILE, \IZON\DB\Files\Types\FileType::class);
//        }
//        $connection->getDatabasePlatform()->registerDoctrineTypeMapping('json', FileType::FILE);
//        if( !\Doctrine\DBAL\Types\Type::hasType(\IZON\DB\Files\Types\RasterImageType::RASTER_IMAGE) ) { // fix multiple app config creation
//            \Doctrine\DBAL\Types\Type::addType(\IZON\DB\Files\Types\RasterImageType::RASTER_IMAGE, \IZON\DB\Files\Types\RasterImageType::class);
//        }
//        $connection->getDatabasePlatform()->registerDoctrineTypeMapping('json', RasterImageType::RASTER_IMAGE);

        foreach ($this->eventSubscribers as $subscriber) {
            $connection->getEventManager()->addEventSubscriber($subscriber);
        }
        $connection->getDatabasePlatform()->registerDoctrineTypeMapping('enum','string');
        return $connection;
    }

    /**
     *
     * @param string|null $proxyDir where to store proxies
     */
    public function setProxyDir(?string $proxyDir) {
        $this->proxyDir = $proxyDir;
    }

    /**
     * @param string $charset charset of connection default is utf8
     */
    public function setCharset(string $charset) {
        $this->charset = $charset;
    }

    /**
     * @param SQLLogger $sqlLogger
     */
    public function setSQLLogger(SQLLogger $sqlLogger) {
        $this->sqlLogger = $sqlLogger;
    }

    /**
     * @param EventSubscriber $subscriber
     */
    public function addEventSubscriber(EventSubscriber $subscriber) {
        $this->eventSubscribers[] = $subscriber;
    }

    public function addBehaviour(BehaviourInterface $behaviour) {
        if(!empty($behaviour->getEventSubscriber())) {
            $this->addEventSubscriber($behaviour->getEventSubscriber());
        }
    }

    /**
     * adds type registrar to register additional types
     * @param TypeRegistrarInterface $typeRegistrator
     */
    public function addTypeRegistrar(TypeRegistrarInterface $typeRegistrator) {
        $this->typeRegistrars[] = $typeRegistrator;
    }

    /**
     * @param string $typeName
     * @param string $typeClass
     */
    public function registerType(string $typeName, string $typeClass) {
        $this->typeRegistrars[] = new SimpleTypeRegistrar($typeName, $typeClass);
    }
}
