<?php

namespace IZON\DB\EntityManager\Factories;

use Doctrine\Common\Annotations\AnnotationReader;
use Doctrine\Common\Annotations\CachedReader;
use Doctrine\Common\Cache\ApcuCache;
use Doctrine\Common\Cache\ArrayCache;
use Doctrine\Common\Cache\Cache;
use Doctrine\Common\Cache\FilesystemCache;
use Doctrine\Common\Proxy\AbstractProxyFactory;
use IZON\DB\Behaviour\BehaviourInterface;
use IZON\DB\EntityManager\Configuration;
use Doctrine\ORM\Mapping\NamingStrategy;
use Doctrine\ORM\ORMException;
use Doctrine\ORM\Tools\Setup;
use IZON\DB\ConnectionInterface;
use IZON\DB\Connections\Configuration as DBConnectionConfiguration;
use IZON\DB\Connections\Connection;
use IZON\DB\EntityManager\EntityManager;
use IZON\DB\EntityManagerInterface;
use IZON\DB\Exceptions\DBException;
use IZON\DB\Extensions\FunctionExtensionInterface;
use IZON\DB\Mapping\Driver\AnnotationDriver;
use IZON\DB\Naming\Impl\DefaultNamingStrategy;
use IZON\DB\Repository\BaseRepository;
use IZON\DB\Repository\Factories\AutobindQueryRepositoryFactory;

/**
 * Factory for creating entity manager
 *
 * description of advenced settings used can be used found
 * https://www.doctrine-project.org/projects/doctrine-orm/en/2.6/reference/advanced-configuration.html
 * DOTO: Auto-generating Proxy Classes (OPTIONAL) - production/develop
 *
 * @author Lukáš Linhart
 */
class EntityManagerFactory {

    /**
     * @var ConnectionInterface
     */
    protected $dbConnection;

    /**
     * @var string[]
     */
    protected $entityDirectories = [];

    /**
     * @var bool
     */
    protected $isDevel = false;

    /**
     * @var string class
     */
    protected $defaultRepositoryClassName = BaseRepository::class;

    /**
     * @var NamingStrategy
     */
    protected $namingStrategy = null;

    /**
     * @var FunctionExtensionInterface[]
     */
    protected $functionExtensions = [];
    /**
     * @var string[] class names
     */
    protected $filters = [];
    /**
     * @var array
     */
    protected $filterParameters = [];

    /**
     * @var Cache cache to store metadata describing db entities
     */
    protected $metadataCache;

    /**
     * @var Cache cache to store parsed dql queries
     */
    protected $queryCache;

    /**
     * @var null|Cache cache that stores results of sql queries
     */
    protected $resultCache = null;

    /**
     *
     * @param ConnectionInterface $dbConnection db connection
     * @param array $entityDirectories array of string paths to directories with Entities
     * @param bool $isDevel use devel mode, default false
     * @throws DBException
     */
    public function __construct(
        ConnectionInterface $dbConnection,
        array $entityDirectories,
        bool $isDevel = false
    ) {
        $this->dbConnection = $dbConnection;
        if (empty($entityDirectories)) {
            throw new DBException('No entity dirs specified');
        }
        $this->entityDirectories = $entityDirectories;
        $this->isDevel = $isDevel;
        $this->namingStrategy = new DefaultNamingStrategy(CASE_LOWER);
        $this->metadataCache = new ArrayCache();
        $this->queryCache = new ArrayCache();
    }
    /**
     *
     * @param NamingStrategy $namingStrategy
     */
    public function setNamingStrategy(NamingStrategy $namingStrategy) {
        $this->namingStrategy = $namingStrategy;
    }
    /**
     * method create new entity manager by configuration
     *
     * @return EntityManagerInterface
     *
     * @throws ORMException
     */
    public function create(): EntityManagerInterface {
        // register anotations using composer autoloader
        // DOTO: add posibility to define anotations
        \Doctrine\Common\Annotations\AnnotationRegistry::registerLoader('class_exists');

        // old, remove after testing
//        $config = Setup::createConfiguration($this->isDevel, $proxyDir);

        $config = new Configuration();
        // configure metadata driver
        $config->setMetadataDriverImpl($this->createAnnotationDriver($this->entityDirectories));
        //
        $config->setRepositoryFactory(new AutobindQueryRepositoryFactory());

        $config->setNamingStrategy($this->namingStrategy);

        foreach($this->functionExtensions as $functionExtension) {
            $functionExtension->register($config);
        }

        foreach($this->filters as $filter) {
            $config->addFilter($filter, $filter);
        }

        if( $this->defaultRepositoryClassName !== null ) {
            $config->setDefaultRepositoryClassName($this->defaultRepositoryClassName );
        }

        // configure folder to store proxies
        $proxyDir = null;
        $dbConnectionConfig = $this->dbConnection->getConfiguration();
        if( $dbConnectionConfig instanceof DBConnectionConfiguration
            && $dbConnectionConfig->getProxyDir() !== null ) {
            $proxyDir = $dbConnectionConfig->getProxyDir();
        }

        // caching settings
        $config->setMetadataCacheImpl($this->metadataCache);
        $config->setQueryCacheImpl($this->queryCache);
        if( $this->resultCache !== null ) {
            $config->setResultCacheImpl($this->resultCache);
        }

        $config->setRepositoryProxiesDir($proxyDir .'/Repositories');

        // configure Entity proxies for lazy loading
        $config->setProxyNamespace('DoctrineProxies');
        if( $proxyDir !== null ) {
            // configure entities proxy dir
            $config->setProxyDir($proxyDir .'/Entities');
            if($this->isDevel) {
                // always regenerate proxy
                $config->setAutoGenerateProxyClasses(AbstractProxyFactory::AUTOGENERATE_EVAL);
            } else {
                // generate proxy only if not exist
                $config->setAutoGenerateProxyClasses(AbstractProxyFactory::AUTOGENERATE_FILE_NOT_EXISTS);
            }
        } else {
            // do not generate proxies only eval them
            $config->setAutoGenerateProxyClasses(AbstractProxyFactory::AUTOGENERATE_EVAL);
        }

        // obtaining the entity manager
        $entityManager = EntityManager::create($this->dbConnection, $config, $this->dbConnection->getEventManager());
        foreach($this->filters as $name) {
            $filter = $entityManager->getFilters()->enable($name);
            foreach($this->filterParameters[$name] as $key => $value) {
                $filter->setParameter($key, $value);
            }
        }
        return $entityManager;
    }

    /**
     * @param string $defaultRepositoryClassName class to be used as repository
     */
    function setDefaultRepositoryClassName($defaultRepositoryClassName) {
        $this->defaultRepositoryClassName = $defaultRepositoryClassName;
    }

    /**
     * Adds a new default annotation driver with a correctly configured annotation reader.
     * Notation `@ORM\Entity`is supported.
     *
     * @param array $paths
     * @return AnnotationDriver
     */
    public function createAnnotationDriver($paths = []): AnnotationDriver {
        $reader = new AnnotationReader();
        return new \IZON\DB\Mapping\Driver\AnnotationDriver(
            new CachedReader($reader, new ArrayCache()),
            (array)$paths
        );
    }

    /**
     * @param FunctionExtensionInterface $extension
     */
    public function addFunctionExtension(FunctionExtensionInterface $extension) {
        $this->functionExtensions[] = $extension;
    }

    public function addBehaviour(BehaviourInterface $behaviour) {
        if(!empty($behaviour->getSQLFilter())) {
            $this->filters[] = $behaviour->getSQLFilter();
            $this->filterParameters[$behaviour->getSQLFilter()] = $behaviour->getFilterParameters();
        }
    }

    /**
     * @param Cache $metadataCache cache to store metadata describing db entities
     */
    public function setMetadataCache(Cache $metadataCache): void {
        $this->metadataCache = $metadataCache;
    }

    /**
     * @param Cache $queryCache cache to store parsed dql queries
     */
    public function setQueryCache(Cache $queryCache): void {
        $this->queryCache = $queryCache;
    }

    /**
     * @param Cache $resultCache  cache that stores results of sql queries
     */
    public function setResultCache(Cache $resultCache): void {
        $this->resultCache = $resultCache;
    }


}
