<?php

namespace IZON\DB\Paginator;

use IZON\DB\Exceptions\DBException;
use IZON\DB\QueryConfig\QueryConfigInterface;
use IZON\DB\QueryParams\OrderBy;
use IZON\DB\QueryParams\QueryParams;
use function IZON\String\startsWith;

/**
 * slouzi jako konfigurace strankovani
 *
 * TODO: mozna vyresit inteligentneji nastavovani ostatnich parametru
 */
class PaginatorConfig implements PaginatorConfigInterface {

    /**
     * @var string
     */
    protected $queryName;

    /**
     * @var string[] regexes defining parametes
     */
    protected $parameterDefinitions = [];

    /**
     * @var string[] regexes defining control parameters to use for construction of query, not passed to db
     */
    protected $controlParameterDefinitions = [];

    /**
     * @var string[] regexes defining orders that can be used
     */
    protected $orderDefinitions = [];

    /**
     * @var mixed[]
     */
    protected $parameters = [];

    /**
     * @var mixed[]
     */
    protected $controlParameters = [];

    /**
     * @var OrderBy[]
     */
    protected $orders = [];

    /**
     * @var int
     */
    protected $pageSize = PaginatorConfigInterface::DEFAULT_PAGE_SIZE;

    /**
     * @var int
     */
    protected $firstResult = 0;


    function __construct($queryName, $parameterDefinitions, $controlParameterDefinitions, $orderDefinitions) {
        $this->queryName = $queryName;
        $this->parameterDefinitions = $parameterDefinitions;
        $this->controlParameterDefinitions = $controlParameterDefinitions;
        $this->orderDefinitions = $orderDefinitions;
    }

    function setPageSize(int $pageSize) {
        $this->pageSize = $pageSize;
    }

    public function getPageSize(): int {
        return $this->pageSize;
    }

    public function getQueryName(): string {
        return $this->queryName;
    }

    /**
     * @param string $propertyName
     * @return bool
     */
    public function hasParameter(string $propertyName): bool {
        return array_key_exists($propertyName, $this->parameters)
                || array_key_exists($propertyName, $this->controlParameters);
    }

    /**
     * @param string $paramName
     * @param mixed $paramValue
     * @throws DBException
     */
    public function putParameter(string $paramName, $paramValue) {
        if( !$this->__isset($paramName) ) {
            throw new DBException("Parameter $paramName not defined");
        }
        if( $this->isParameter($paramName) ) {
            $this->parameters[$paramName] = $paramValue;
        } else {
            $this->controlParameters[$paramName] = $paramValue;
        }
    }

    /**
     * @param string $paramName
     * @return mixed
     * @throws DBException
     */
    public function getParameter(string $paramName) {
        if(!$this->__isset($paramName)) {
            throw new DBException("Parameter $paramName not defined");
        }
        if( $this->isParameter($paramName) ) {
            return $this->parameters[$paramName];
        } else {
            return $this->controlParameters[$paramName];
        }
    }

    /**
     * @param string $paramName
     * @throws DBException
     */
    public function removeParameter(string $paramName) {
        if(!$this->__isset($paramName)) {
            throw new DBException("Parameter $paramName not defined");
        }
        if( $this->isParameter($paramName) ) {
            unset($this->parameters[$paramName]);
        } else {
            unset($this->controlParameters[$paramName]);
        }

    }

    /**
     * @return array pole se vsemi nastavenymi parametry
     */
    public function getParameters(): array {
        $parameters = [];
        foreach($this->parameters as $propertyName => $value) {
            if( $this->__isset($propertyName) ) {
                $parameters[$propertyName] = $value;
            }
        }
        foreach($this->controlParameters as $propertyName => $value) {
            if( $this->__isset($propertyName) ) {
                $parameters[$propertyName] = $value;
            }
        }
        return $parameters;
    }

    /**
     *
     * @param string $propertyName
     * @return boolean
     * @throws DBException
     */
    public function __isset(string $propertyName): bool {
        return $this->isParameter($propertyName)
                || $this->isControlParameter($propertyName);
    }

    protected function isParameter(string $paramName): bool {
        foreach($this->parameterDefinitions as $definedFieldName) {
            $result = preg_match("#^" . $definedFieldName . "$#u", $paramName);
            if($result === false) {
                throw new DBException("Can't match deffined field " . $definedFieldName . " pattern");
            } else if($result) {
                return true;
            }
        }
        return false;
    }

    protected function isControlParameter(string $paramName): bool {
        foreach($this->controlParameterDefinitions as $definedFieldName) {
            $result = preg_match("#^" . $definedFieldName . "$#u", $paramName);
            if($result === false) {
                throw new DBException("Can't match defined field " . $definedFieldName . " pattern");
            } else if($result) {
                return true;
            }
        }
        return false;
    }

    /**
     * To enable calling get*() and set*().
     * For setting properties using getters and setters.
     *
     * @param string $methodName
     * @param array $arguments
     * @throws DBException If property is not defined.
     */
    public function __call($methodName, array $arguments) {
        if( !startsWith($methodName, 'get')
            && !startsWith($methodName, 'set') ) {
            throw new DBException('method '. $methodName ." is not supported. Only getters and setters are supported");
        }

        $propertyNamePart = substr($methodName, 3); # Removes first three characters ('get' or 'set')
        $propertyName = lcfirst($propertyNamePart); # Lower case first letter

        # Getter
        if( startsWith($methodName, 'get') ) {
            # If field is defined
            if($this->__isset($propertyName)) {
                return $this->getParameter($propertyName);
            } else {
                throw new DBException("Property $propertyName not defined");
            }
        }

        # Setter
        if(startsWith($methodName, 'set')) {
            # If field is defined
            if( $this->__isset($propertyName) ) {
                $value = $arguments[0]; # Get value
                $this->putParameter($propertyName, $value);
            } else {
                throw new DBException("Property $propertyName not defined");
            }
        }

    }


    /**
     * @return OrderBy[]
     */
    public function getOrders(): array {
        return $this->orders;
    }
    /**

     * Sets ordering by given property and given direction.
     *
     * @param string $propertyName
     * @param string $direction OrderBy::ORDER_ASC|OrderBy::ORDER_SESC
     * @throws DBException
     */
    public function addOrder(string $orderName, string $direction) {
        if(!$this->isOrder($orderName)) {
            throw new DBException("Property '{$orderName}' is not in defined orders and therefore cannot be used for orders.");
        }
        $orderBy = new OrderBy($orderName, $direction);
        $this->orders[] = $orderBy;
    }

    /**
     * @param string $propertyName
     * @return bool
     */
    public function isOrder(string $orderName): bool {
        foreach($this->orderDefinitions as $orderDefinition) {
            $result = preg_match("#^" . $orderDefinition . "$#u", $orderName);
            if($result === false) {
                throw new DBException("Can't match to deffined order " . $orderDefinition . " pattern");
            } else if($result) {
                return true;
            }
        }
        return false;
    }

    public function clearOrders() {
        $this->orders = [];
    }

    /**
     * @return int
     */
    public function getFirstResult(): int {
        return $this->firstResult;
    }

    /**
     * @param int $firstResult
     */
    public function setFirstResult(int $firstResult) {
        $this->firstResult = $firstResult;
    }

    /**
     * {@inheritedDoc}
     */
    public function toQueryParams(): QueryParams {
        $queryParams = new QueryParams($this->parameters, $this->controlParameters, $this->orders);
        return $queryParams;
    }
}
