<?php

namespace IZON\DB\Utils;

use Exception;
use IZON\DB\DBObject;
use IZON\DB\Definitions\AbstractDefinition;
use IZON\DB\Impl\DBConnectionCommon;
use IZON\DB\NamingStrategy;
use IZON\DB\QueryFactory;
use IZON\DB\QueryHelpers;
use IZON\DB\Utils\PropertyDescriptions\AbstractPropertyDescription;
use IZON\DB\Utils\PropertyDescriptions\FileDescription;
use IZON\DB\Utils\PropertyDescriptions\ImageDescription;
use IZON\IO\File;
use IZON\IO\Image;
use IZON\Logs\Logger;
use IZON\Utils\Date;
use PhpDocReader\AnnotationException;
use PhpDocReader\PhpDocReader;
use Psr\Log\LoggerInterface;
use ReflectionClass;
use ReflectionNamedType;
use ReflectionProperty;

/**
 * Obsahuje popis tridy potrebny pro vytvoreni dotazu
 * a spetne naplneni objektu a podobjedku a kolekci
 */
class ClassDescription
{
    /**
     * nacita datovy typ z dokumentacniho bloku
     */
    protected static ?PhpDocReader $docReader = null;
    /**
     * skalarni typy promenych v php
     * http://php.net/manual/en/language.types.intro.php
     * @var string[]
     */
    protected static array $scalarTypes = [
        'bool',
        'boolean',
        'string',
        'int',
        'integer',
        'float',
        'double',
        'number' //  integer or float
    ];

    /**
     * @var array<string, class-string> map of guessable classes for scalar types
     */
    protected static array $guessableClasses = [
        'Date' => Date::class,
        'File' => File::class,
        'Image' => Image::class
    ];

    /**
     * pripojeni k db
     * @var DBConnectionCommon
     */
    protected DBConnectionCommon $dbConnection;

    /**
     * jak se maji prekladat nazvy
     * @var NamingStrategy
     */
    protected NamingStrategy $namingStrategy;

    /**
     * nazev tridy, kterou tento objekt popisuje
     * @var class-string
     */
    protected string $className;

    /**
     * @var ReflectionClass
     */
    protected ReflectionClass $classReflectionClass;

    /**
     * @var ReflectionProperty[] hash obsahujici ReflectionProperty k jmenu property
     */
    protected array $reflectionProperties;

    /**
     * jmeno tabulky na kterou se ma mapovat
     * @var string
     */
    protected string $tableName;

    /**
     * descriptions of properties of object
     * subclasses of AbstractPropertyDescription
     * @var AbstractPropertyDescription[]
     */
    protected array $propertyDescriptions = [];

    /**
     * preklad jmen properties na odpovidajici jmena sloupcu
     * @var array
     */
    protected array $propertiesToColumnsTranslation = [];

    protected array $propertiesToColumnsStoreTranslation = [];

    /**
     * jak se jmenuje property identifikujici objekt, defaultne je to id
     * @var string
     */
    protected string $idPropertyName = "id";

    protected LoggerInterface $log;

    /**
     *
     * @param DBConnectionCommon $dbConnection
     * @param class-string $className
     * @throws Exception
     */
    public function __construct(DBConnectionCommon $dbConnection, string $className)
    {
        // sprovoznit logovani
        $this->log = Logger::getLogger(self::class);

        if (!class_exists($className)) { // tests if class is DBObject
            throw new Exception("Class $className does exist.");
        }

        if (!is_subclass_of($className, DBObject::class)) { // tests if class is DBObject
            throw new Exception("$className does not implement " . DBObject::class);
        }

        $this->dbConnection = $dbConnection;

        $this->namingStrategy = $dbConnection->getNamingStrategy();

        $this->classReflectionClass = new ReflectionClass($className);
        $this->className = $className;

        // nastaveni jmena tabulky
        if (method_exists($this->className, "__tableName")) {
            $this->tableName = $this->className::__tableName();
        } else {
            $this->tableName = $this->namingStrategy->classToTableName($this->className);
        }
        // nastaveni prekladu jmena property na jmeno sloupce
        $this->initProperities();

        // TODO: test jestli query factory vyhovuje parametrum
    }

    /**
     * intializes all properties of specified class
     */
    protected function initProperities()
    {
        // go thought all properties
        $propertiesHash = [];
        foreach ($this->classReflectionClass->getProperties() as $property) {
            $propertiesHash[$property->getName()] = $property;
        }
        //go thought all parents properties
        $reflectionClass = $this->classReflectionClass;
        while ($reflectionClass->getParentClass()) {
            $reflectionClass = $reflectionClass->getParentClass();
            foreach ($reflectionClass->getProperties() as $property) {
                if (!isset($propertiesHash[$property->getName()])) {
                    $propertiesHash[$property->getName()] = $property;
                }
            }
        }
        $this->reflectionProperties = $propertiesHash;

        // definice jednotlivych
        $propertyDefinitions = [];
        $propertyTypes = [];
        foreach ($propertiesHash as $property) {
            // vsechny nastavit aby byly pristupne
            $property->setAccessible(true);

            // zjstit jentli ma definici k vride
            $propertyDefinition = null;
            if ($this->classReflectionClass->hasMethod("__" . $property->getName())) {
                $method = $this->classReflectionClass->getMethod("__" . $property->getName());
                $propertyDefinition = $method->invoke(null);
            }

            $propertyDefinitions[$property->getName()] = $propertyDefinition;
            $propertyType = $this->getCommentDataType($property);
            if ($propertyType === null) {
                $propertyType = $this->getTypeDefinitionDataType($property);
            }
            $propertyTypes[$property->getName()] = $propertyType;
        }

        $this->propertyDescriptions = $this->getPropertyDescriptions($propertyDefinitions, $propertyTypes);

        foreach ($this->propertyDescriptions as $key => $propertyDescription) {
            // naplni preklady properit na soubory
            $this->propertiesToColumnsTranslation[$key] = $propertyDescription->getColumnName();

            if (!$propertyDescription->isReadonly()) {
                $this->propertiesToColumnsStoreTranslation[$key] = $propertyDescription->getColumnName();
            }
        }
    }

    /**
     * vytahne datovy typ z komentare u property v mapovane tride
     * @param ReflectionProperty $property
     * @return string|null
     * @throws AnnotationException
     */
    protected function getCommentDataType(ReflectionProperty $property): ?string
    {
        $reader = self::getDocReader();
        //find existing class
        $dataType = $reader->getPropertyClass($property); // try to find class

        if (is_string($dataType) && !empty($dataType)) {
            return $dataType;
        }

        // no existing class -> try to find scalar type
        $foundVarType = preg_match('/@var\s+([^\s]+)/', $property->getDocComment(), $matches);
        if ($dataType == null && !$foundVarType) {
            return null;
        }

        // nacte skalarni typ
        list(, $varType) = $matches;

        $compositeDataTypes = explode('|', $varType);

        foreach ($compositeDataTypes as $compositeDataType) {
            if (isset(self::$guessableClasses[$compositeDataType])) { // try to guess class
                return self::$guessableClasses[$compositeDataType];
            }
            if (in_array($compositeDataType, self::$scalarTypes)) {  // use first recognised scalar data type
                return $compositeDataType;
            }
        }

        return null;
    }

    /**
     * @return PhpDocReader
     */
    protected function getDocReader(): PhpDocReader
    {
        //TODO: set new phpDocReader
        if (self::$docReader === null) {
            self::$docReader = new PhpDocReader();
        }
        return self::$docReader;
    }

    /**
     * returns strict data type if defined
     * @param ReflectionProperty $property
     * @return string|null
     */
    protected function getTypeDefinitionDataType(ReflectionProperty $property): ?string
    {
        // find type for property strict type
        $reflectionType = $property->getType();
        if ($reflectionType instanceof ReflectionNamedType) {
            if (
                in_array($reflectionType->getName(), self::$guessableClasses)
                || in_array($reflectionType->getName(), self::$scalarTypes)
            ) {  //is scalar type
                return $reflectionType->getName();
            }
        }
        return null;
    }

    /**
     *
     * @param AbstractDefinition[] $propertyDefinitions
     * @param string[] $propertyTypes classes and types of properties
     * @return AbstractPropertyDescription[]
     */
    protected function getPropertyDescriptions(array $propertyDefinitions, array $propertyTypes)
    {
        return $this->dbConnection->getPropertyDescriptions($this, $propertyDefinitions, $propertyTypes);
    }

    public function getColumnName($propertyName)
    {
        if (!isset($this->propertiesToColumnsTranslation[$propertyName])) {
            throw new Exception("Trida " . $this->className . " nema property '" . $propertyName . "'");
        }
        return $this->propertiesToColumnsTranslation[$propertyName];
    }

    public function getClassName()
    {
        return $this->className;
    }

    /**
     * vrati nazev tabulky pro dany objekt
     * @return string
     */
    public function getTableName()
    {
        return $this->tableName;
    }

    /**
     * vrati preklad vsech poperit objektu na jmena sloupcu
     * @return array
     */
    public function getPropertiesToColumnsTranslation()
    {
        return $this->propertiesToColumnsTranslation;
    }

    /**
     * @return array
     */
    public function getPropertiesToColumnsStoreTranslation()
    {
        return $this->propertiesToColumnsStoreTranslation;
    }

    /**
     * nazev property, ktera obsahuje id objektu
     * @return string
     */
    public function getIdPropertyName()
    {
        return $this->idPropertyName;
    }

    /**
     * nazev sloupce, ktery obsahuje id
     * @return string
     */
    public function getIdColumnName()
    {
        return $this->propertiesToColumnsTranslation[$this->idPropertyName];
    }

    /**
     * returns value of property with name $propertyName from $object
     * @param object $object
     * @param string $propertyName
     * @return mixed Description
     */
    public function getPropertyValue($object, $propertyName)
    {
        if (!isset($this->reflectionProperties[$propertyName])) {
            throw new Exception("Trida " . $this->className . " nema property " . $propertyName);
        }

        $reflectionProperty = $this->reflectionProperties[$propertyName];
        //        $reflectionProperty->setAccessible(true);
        return $reflectionProperty->getValue($object);
    }

    /**
     *
     * @param object $object object property to be set to
     * @param string $propertyName name of the property to be set
     * @param mixed $propertyValue value of property to be set
     */
    public function setPropertyValue($object, $propertyName, $propertyValue)
    {
        if (!isset($this->reflectionProperties[$propertyName])) {
            throw new Exception("Trida " . $this->className . " nema property " . $propertyName);
        }

        $reflectionProperty = $this->reflectionProperties[$propertyName];
        $reflectionProperty->setValue($object, $propertyValue);
    }

    /**
     *
     * @param QueryFactory|array|null $qureyFactory
     * @param string $methodName
     * @param integer $parametersCount
     * @param QueryHelpers $queryHelpers
     * @return string
     * @throws Exception
     */
    public function getQueryString($qureyFactory, $methodName, $parametersCount, QueryHelpers $queryHelpers)
    {
        //        var_dump($qureyFactory);
        $factoryMethodName = $methodName . "_" . $parametersCount;
        if (!is_array($qureyFactory)) {
            //            var_dump($qureyFactory);
            if ($qureyFactory === null) {
                throw new Exception("Pro tridu " . $this->className . " neexistuje query factory");
            }
            if (!method_exists($qureyFactory, $factoryMethodName)) {
                throw new Exception("Pro tridu " . $this->className . " neexistuje metoda $factoryMethodName pro vytvoreni dotazu.");
            }

            $queryString = $qureyFactory->$factoryMethodName($queryHelpers);
        } else {
            //            var_dump($qureyFactory[$factoryMethodName]);
            if (!isset($qureyFactory[$factoryMethodName]["queryFactory"])) {
                throw new Exception("Pro tridu " . $this->className . " neexistuje metoda $factoryMethodName pro vytvoreni dotazu.");
            }
            $factoryCallable = $qureyFactory[$factoryMethodName]["queryFactory"];
            $queryString = $factoryCallable($queryHelpers);
        }
        return $queryString;
    }

    /**
     *
     * @param QueryFactory|array|null $qureyFactory
     * @param string $methodName
     * @param array $parameters
     * @param array $queryTypeParameters
     * @param QueryHelpers $queryHelpers
     * @return string
     * @throws Exception
     */
    public function getCustomQueryString($qureyFactory, $methodName, array $parameters, array $queryTypeParameters, QueryHelpers $queryHelpers)
    {
        $factoryMethodName = $methodName;
        if (!is_array($qureyFactory)
            && $qureyFactory instanceof QueryFactory) {
            if (!method_exists($qureyFactory, $factoryMethodName)) {
                throw new Exception("Pro tridu " . $this->className . " neexistuje metoda $methodName pro vytvoreni dotazu.");
            }
            return $qureyFactory->$factoryMethodName($queryHelpers, $parameters, $queryTypeParameters);
        } elseif (is_array($qureyFactory)) {
            //            var_dump($qureyFactory[$factoryMethodName]);
            if (!isset($qureyFactory[$factoryMethodName]["queryFactory"])) {
                throw new Exception("Pro tridu " . $this->className . " neexistuje metoda $methodName pro vytvoreni dotazu.");
            }
            $factoryCallable = $qureyFactory[$factoryMethodName]["queryFactory"];
            return $factoryCallable($queryHelpers, $parameters, $queryTypeParameters);
        } else {
            throw new Exception("Neni ani QueryFactory ani pole s query factory");
        }
    }

    /**
     * @param string $propertyName
     * @returns boolean true if property is file
     */
    public function isPropertyFile($propertyName)
    {
        $propertyDescription = $this->getPropertyDescription($propertyName);
        return $propertyDescription instanceof FileDescription
            || $propertyDescription instanceof ImageDescription; // @phpstan-ignore instanceof.alwaysFalse
    }

    /**
     *
     * @param string $propertyName
     * @return AbstractPropertyDescription
     * @throws Exception
     */
    public function getPropertyDescription($propertyName)
    {
        if (!isset($this->propertyDescriptions[$propertyName])) {
            throw new Exception("Popis $propertyName neni definovan.");
        }
        return $this->propertyDescriptions[$propertyName];
    }
}
