<?php

namespace IZON\DB;

use IZON\DB\Impl\DBConnectionCommon;
use IZON\DI\FactoryObject;
use IZON\Logs\Logger;
use Laminas\Code\Generator\ClassGenerator;
use Laminas\Code\Generator\MethodGenerator;
use Laminas\Code\Generator\ParameterGenerator;
use Laminas\Code\Generator\PropertyGenerator;
use Laminas\Code\Reflection\ClassReflection;
use ReflectionNamedType;

/**
 * Slouzi k vytvoreni daa
 */
class TransactionManagementFactoryObject implements FactoryObject
{
    protected DBConnectionCommon $connection;

    /**
     * na jaky objekt se maji preposilat pozadavky
     * @var object
     */
    protected $proxyObject;

    private Logger $log;


    public function __construct(DBConnectionCommon $connection, $proxyObject)
    {
        $this->connection = $connection;
        $this->proxyObject = $proxyObject;

        $this->log = Logger::getLogger(self::class);
    }


    public function getObject()
    {
        $objectReflectionClass = new ClassReflection($this->proxyObject); // class Foo of namespace A

        $objectName = $objectReflectionClass->getName() . "TxProxy";

        // trida jeste neexistuje je ji treba vytvorit a ninicializovat
        if (!class_exists($objectName)) {
            $implCacheFileName = $this->connection->getTransactionProxyImplCacheDir() . "/" . str_replace('\\', '/', $objectName) . ".php";
            $fileLastModified = filemtime($objectReflectionClass->getFileName());

            if (
                $this->connection->getTransactionProxyImplCacheDir() != null // je definovan adresar kam cachovat daa
                && file_exists($implCacheFileName) // uz existuje vygenerovane dao
                && $fileLastModified <= filemtime($implCacheFileName) // cache je novejsi nez soubor ve kterem byl vygenerovan
            ) {
                // vem dao z cache
                require $implCacheFileName;
            } else {
                // TODO: pridat cashovani vytvorene tridy
                $parentClass = "\\" . get_class($this->proxyObject);

                $parentInterfaces = $objectReflectionClass->getInterfaceNames();
                $implInterfaces = [];
                foreach ($objectReflectionClass->getInterfaces() as $inedx => $interface) {
                    $implInterfaces[$inedx] = "\\" . $interface->name;
                }

                $class = new ClassGenerator(
                    $objectName,
                    $objectReflectionClass->getNamespaceName(),
                    null,
                    $parentClass,
                    $implInterfaces
                );

                $class->addProperty("connection", null, PropertyGenerator::FLAG_PRIVATE);
                $class->addProperty("proxyObject", null, PropertyGenerator::FLAG_PRIVATE);

                // vytvori konstruktor pro predani parametru
                $method = new MethodGenerator(
                    "__construct",
                    [new ParameterGenerator("connection", "\IZON\DB\DBConnection"), "proxyObject"]
                );
                $method->setBody('$this->connection = $connection; $this->proxyObject = $proxyObject;');
                $class->addMethodFromGenerator($method);

                // prochaci metody rozhrani daa
                foreach ($objectReflectionClass->getMethods() as $method) {
                    if ($method->name == '__construct') {
                        continue;
                    }

                    $methodGenerator = MethodGenerator::copyMethodSignature($method);

                    $paramsNames = [];
                    foreach ($method->getParameters() as $parameter) {
                        $methodGenerator->setParameter(ParameterGenerator::fromReflection($parameter));
                        $paramsNames[] = '$' . $parameter->getName();
                    }

                    /** @var null|ReflectionNamedType $returnType */
                    $returnType = $method->getReturnType();

                    $methodGenerator->setBody(
                        ' $pdo = $this->connection->getPDO(); '
                        . ' $pdo->beginTransaction(); '
                        . ' try { '
                        . ' $ret = $this->proxyObject->' . $method->getName() . '(' . implode(", ", $paramsNames) . ');'
                        . ' $pdo->commit(); '
                        . (($returnType !== null ? $returnType->getName() : null) !== 'void' ? ' return $ret; ' : '')
                        . ' } catch (\Throwable $e) {'
                        . ' $pdo->rollBack(); '
                        . ' throw $e; '
                        . ' }'
                    );
                    $class->addMethodFromGenerator($methodGenerator);
                }

                $generated = $class->generate();

                $this->log->info("Generated class: " . $generated);

                eval($generated);

                if ($this->connection->getTransactionProxyImplCacheDir() !== null) { // je nastaven adresar pro cache
                    // zapsat do cache
                    $implCacheDirName = dirname($implCacheFileName);
                    if (!file_exists($implCacheDirName)) { // neni soubor do ktereho se ma zapsat
                        if (!@mkdir($implCacheDirName, 0777, true)) { // nevyhazovat warning pokud jiz adresar existuje
                            $this->log->warning(
                                "Nepodarilo se vytvorit adresar $implCacheDirName pro cache transaction proxy $objectName"
                            );
                        }
                    }
                    if (!file_put_contents($implCacheFileName, "<?php\r\n" . $generated, LOCK_EX)) {
                        $this->log->warning(
                            "Nepodarilo se zapsat cache $objectName transaction proxy souboru $implCacheFileName"
                        );
                    }
                    if (file_exists($implCacheFileName)) {
                        chmod($implCacheFileName, 0777);
                    }
                }
            }
        }

        $object = new $objectName($this->connection, $this->proxyObject);
        return $object;
    }
}
