<?php

namespace IZON\DB\Paginator;

use Exception;
use IZON\DB\OrderBy;
use IZON\Utils\StringUtils;

/**
 * slouzi jako konfigurace strankovani
 *
 * TODO: mozna vyresit inteligentneji nastavovani ostatnich parametru
 */
class PaginatorConfig
{
    /**
     * jak se jmenuje defaultni strankovac pro danou tridu
     */
    public const DEFAULT_PAGINATOR_NAME = "_default";


    /**
     * defaultni velikost stranky
     */
    public const DEFAULT_MAX_PAGE_SIZE = 100;

    /**
     * @var null|string jak se jmenuje strankovac, ktery ma tuto stranku obslouzit
     */
    protected ?string $paginatorName = null;

    /**
     * pocet zaznamu na stranku
     */
    protected $maxPageSize;

    /**
     * @var integer index of first result
     */
    protected $firstResult = 0;

    /**
     * @var null|string[] jaka pole jsou definovana pro tento strankovac
     */
    protected $definedFields = null;

    /**
     *
     * @var null|string[] podle kterych poli je mozno radit
     */
    protected $definedOrders = null;

    /**
     * @var array<string, mixed> parametry strankovace
     */
    protected array $parameters = [];


    /**
     * jednotlive sloupce podle kterych se ma radit ['propertyPodleKtereSeMaRadit', asc/desc]
     * @var array<int, array{0: string, 1: OrderBy::ASC|OrderBy::DESC}> pole razeni
     */
    protected array $orders = [];

    /**
     *
     * @param string $paginatorName
     * @param string[] $definedFields pole, ktera se mohou nastavit z formulare nebo jinym zpusobem z requestu
     * @param int $maxPageSize
     */
    public function __construct(
        string $paginatorName,
        array $definedFields = [],
        $definedOrders = [],
        int $maxPageSize = self::DEFAULT_MAX_PAGE_SIZE
    ) {
        $this->paginatorName = $paginatorName;
        $this->definedFields = $definedFields;
        $this->definedOrders = $definedOrders;
        $this->maxPageSize = $maxPageSize;

        if (!is_array($definedOrders)) { // backward compatibility hack
            $this->maxPageSize = $definedOrders;
            $this->definedOrders = [];
        }
    }

    public function getPaginatorName()
    {
        return $this->paginatorName;
    }

    public function getMaxPageSize()
    {
        return $this->maxPageSize;
    }

    public function setMaxPageSize($maxPageSize)
    {
        $this->maxPageSize = $maxPageSize;
    }

    public function getFirstResult()
    {
        return $this->firstResult;
    }

    public function setFirstResult($firstResult)
    {
        $this->firstResult = $firstResult;
    }

    public function getDefinedFields()
    {
        return $this->definedFields;
    }


    /**
     * @return array<int, array{0: string, 1: OrderBy::ASC|OrderBy::DESC}>
     */
    public function getOrders(): array
    {
        return $this->orders;
    }

    /**
     * @param array<int, array{0: string, 1: OrderBy::ASC|OrderBy::DESC}> $orders
     * @return void
     */
    public function setOrders(array $orders): void
    {
        $this->orders = $orders;
    }

    /**
     *
     * @param string $propertyName na jake propery
     * @param OrderBy::ASC|OrderBy::DESC $asc jestli ma byt asc
     */
    public function addOrder(string $propertyName, bool $asc = true)
    {
        $this->orders[] = [$propertyName, $asc];
    }

    public function clearOrders()
    {
        $this->orders = [];
    }

    public function isOrderable($propertyName)
    {
        return in_array($propertyName, $this->definedOrders);
    }

    /**
     *
     * @param string $propertyName
     * @return boolean
     */
    public function hasParameter($propertyName)
    {
        return isset($this->parameters[$propertyName]);
    }

    public function putParameter($paramName, $paramValue)
    {
        if (!$this->__isset($paramName)) {
            throw new Exception("Property $paramName not defined");
        }
        $this->parameters[$paramName] = $paramValue;
    }

    public function __isset($propertyName)
    {
        foreach ($this->definedFields as $definedFieldName) {
            $result = preg_match("#^" . $definedFieldName . "$#", $propertyName);
            if ($result === false) {
                throw new Exception("Can't match defined field " . $propertyName . " pattern");
            } elseif ($result) {
                return true;
            }
        }
        return false;
    }

    public function removeParameter($paramName)
    {
        if (!$this->__isset($paramName)) {
            throw new Exception("Property $paramName not defined");
        }
        unset($this->parameters[$paramName]);
    }

    /**
     * @return array pole se vsemi nastavenymi parametry
     */
    public function getParameters()
    {
        $params = [];
        foreach ($this->parameters as $propertyName => $value) {
            if ($this->__isset($propertyName)
                && $value !== null) {
                $params[$propertyName] = $value;
            }
        }

        //        var_dump($params);
        return $params;
    }

    public function __get(string $propertyName)
    {
        if (!$this->__isset($propertyName)) {
            throw new Exception("Property $propertyName not defined");
        }
        return $this->getParameter($propertyName);
    }

    public function __set(string $propertyName, $value)
    {
        //        var_dump($propertyName);
        if (!$this->__isset($propertyName)) {
            throw new Exception("Property $propertyName not defined");
        }
        $this->parameters[$propertyName] = $value;
    }

    public function getParameter($paramName)
    {
        if (!$this->__isset($paramName)) {
            throw new Exception("Property $paramName not defined");
        }
        if (!array_key_exists($paramName, $this->parameters)) {
            return null;
        }
        return $this->parameters[$paramName];
    }

    /**
     * aby se dalo volat get[$name]
     * @param string $name
     * @param array $arguments
     */
    public function __call($name, array $arguments)
    {
        if (StringUtils::startsWith($name, "get")) { // obsluhuji se jen get metody
            $propertyName = mb_substr($name, 3); // odstrani get
            $propertyName = lcfirst($propertyName); // prvni znak preved na mala pismena

            if (!$this->__isset($propertyName)) { // talove pole není definovano
                throw new Exception("Property $propertyName not defined");
            }
            return $this->getParameter($propertyName);
        } elseif (StringUtils::startsWith($name, "set")) { // obsluhuji se jen get metody
            $propertyName = mb_substr($name, 3); // odstrani set
            $propertyName = lcfirst($propertyName); // prvni znak preved na mala pismena

            if ($this->__isset($propertyName)) { // talove pole je definovano
                // ziskal argument pro nastaveni
                $firstKey = reset($arguments);
                // nastav hodnotu
                $this->parameters[$propertyName] = $firstKey;
            } else {
                throw new Exception("Property $propertyName not defined");
            }
        }
    }

    public function getDefinedOrders()
    {
        return $this->definedOrders;
    }
}
