<?php

namespace IZON\DB\Paginator;

use Exception;
use IZON\DB\Impl\DBConnectionCommon;
use IZON\DB\QueryHelpers;

/**
 * paginator facroty, ktera se pouzije pro dotazy vytvorene rucne
 */
class CustomPaginatorFactory implements PaginatorFactory
{
    /**
     * @var string name of paginator
     */
    protected $paginatorName;

    /**
     * @var string class paginator factory is for
     */
    protected $class;

    /**
     * @var array property, podle kterych je mozne vyhledavat
     */
    protected $propertes = [];

    /**
     * @var string[]|null podle kterych property je mozne radit
     */
    protected $orders = null;

    /**
     * @var array jake je defaultni razeni
     */
    protected $defaultOrder = ["id", false];

    /**
     * @var callable|null factory pro vytvoreni dotazu obsahujici stranku
     */
    protected $pageQueryFactory = null;

    /**
     * @var callable|null factory pro vytvoreni dotazu na celkovy pocet zaznamu
     */
    protected $countQueryFactory = null;

    /**
     * @var array|null konfugurace ziskana z konfiguracniho souboru
     */
    protected $config;


    public function __construct($paginatorName, $class, array $config = [])
    {
        $this->paginatorName = $paginatorName;
        $this->class = $class;
        $this->config = $config;
    }

    /**
     * provede inicializaci sptankovace
     * @param DBConnectionCommon $connection
     */
    public function init(DBConnectionCommon $connection)
    {
        if (is_null($this->config)) {// konfig uz byl parsovan, neni potreba delat znova
            return;
        }
        $config = $this->config;

        if (!isset($config["properties"])) {
            throw new Exception("V konfiguraci neni definovano podle kterych parametru je mozne vyhledavat");
        }
        if (!isset($config["orders"])) {
            throw new Exception("V konfiguraci neni definovano podle kterych properit je mozne radit");
        }
        if (!isset($config["countQueryFactory"])) {
            throw new Exception("V konfiguraci neni factory metoda countQueryFactory pro vytvoreni count dotazu");
        }
        if (!isset($config["pageQueryFactory"])) {
            throw new Exception("V konfiguraci neni factory  metoda pageQueryFactory pro vytvoreni paget dotazu");
        }

        $this->propertes = $config["properties"];
        $this->orders = $config["orders"];

        $this->countQueryFactory = $config["countQueryFactory"];
        $this->pageQueryFactory = $config["pageQueryFactory"];

        if (isset($config["defaultOrder"])) {
            // TODO: kontrola ze ma spravny tvar
            $this->defaultOrder = $config["defaultOrder"];
        }

        $this->config = null;
    }

    /**
     * vytvori paginator config
     */
    public function createPaginatorConfig($maxPageSize)
    {
        $config = new PaginatorConfig($this->paginatorName, array_keys($this->propertes), $this->orders, $maxPageSize);
        if ($this->defaultOrder) { // default order is defined
            $config->addOrder($this->defaultOrder[0], $this->defaultOrder[1]);
        }
        return $config;
    }

    /**
     *
     * @param QueryHelpers $queryHelpers
     * @param PaginatorConfig $config
     * @return string
     */
    public function page(QueryHelpers $queryHelpers, PaginatorConfig $config)
    {
        $factoryMethod = $this->pageQueryFactory;
        $query = $factoryMethod($queryHelpers, $config);

        return $query;
    }

    /**
     *
     * @param QueryHelpers $queryHelpers
     * @param PaginatorConfig $config
     * @return string
     */
    public function count(QueryHelpers $queryHelpers, PaginatorConfig $config)
    {
        $factoryMethod = $this->countQueryFactory;
        $query = $factoryMethod($queryHelpers, $config);

        return $query;
    }
}
