<?php

namespace IZON\DB\Impl;

use Exception;
use IZON\DB\Dao;
use IZON\DB\DBObject;
use IZON\DB\Paginator\CustomPaginatorFactory;
use IZON\DB\Paginator\DefaultPaginatorFactory;
use IZON\DB\Paginator\PageContent;
use IZON\DB\Paginator\PaginatorConfig;
use IZON\DB\QueryFactory;
use IZON\Logs\Logger;
use IZON\Utils\StringUtils;
use Psr\Log\LoggerInterface;
use ReflectionClass;
use function IZON\Arrays\isEmpty;

/**
 * @template T of DBObject
 * @implements Dao<T>
 */
class DaoImpl implements Dao
{
    protected DBConnectionCommon $dbConnection;

    /**
     * @var ReflectionClass reflection class
     */
    protected ReflectionClass $daoInterfaceReflectionClass;

    /**
     * domenova trida, ktera se ma vracet
     * @var string
     */
    protected string $domainClassName;

    /**
     * query factory, ktera slouzi v vhytvareni fotazu pro jednotlive metody daa
     * @var QueryFactory|array|null
     */
    protected $queryFactory = null;

    /**
     * strankovace, ktere se mohou pouzit pro $domainClassName
     * @var array<string, DefaultPaginatorFactory|CustomPaginatorFactory>
     */
    protected array $paginatorFactories = [];

    private LoggerInterface $log;


    public function __construct(
        DBConnectionCommon $dbConnection,
        $daoInterface,
        ReflectionClass $daoInterfaceReflectionClass,
        $queryFactoriesFiles,
        $paginatorQueryFactoriesFiles
    ) {
        $this->dbConnection = $dbConnection;
        $this->daoInterfaceReflectionClass = $daoInterfaceReflectionClass;
        $this->domainClassName = $daoInterface::DOMAIN_CLASS;

        //        $definingFileDir = dirname($daoInterfaceReflectionClass->getFileName());
        //        $queryFactoryConfig = dirname($definingFileDir) ."/DB/". $daoInterfaceReflectionClass->getShortName() .".php";

        $queryConfig = [];
        foreach ($queryFactoriesFiles as $fileName) {
            $queryConfig = array_merge($queryConfig, require $fileName);
        }
        if (!isEmpty($queryConfig)) {
            $this->queryFactory = $queryConfig;
        } else {
            // nacte potrebnou query factory podle domain class
            $queryFactoryName = str_replace("\\Domain\\", "\\DB\\", $this->domainClassName) . "QueryFactory" . $this->dbConnection->getDbType();
            if (class_exists($queryFactoryName)) {
                $this->queryFactory = new $queryFactoryName();
            } else {
                $queryFactoryName = str_replace("\\Domain\\", "\\DB\\", $this->domainClassName) . "QueryFactory";
                //               var_dump($queryFactoryName);
                if (class_exists($queryFactoryName)) {
                    $this->queryFactory = new $queryFactoryName();
                    //                   var_dump(new $queryFactoryName());
                }
            }
        }

        // zpracuje paginator factory
        $paginatorConfig = [];
        foreach ($paginatorQueryFactoriesFiles as $fileName) {
            //            var_dump($fileName);
            $paginatorConfig = array_merge($paginatorConfig, require $fileName);
        }
        if (!isEmpty($paginatorConfig)) { // mame konfig
            $defaultPaginatorConfig = [];
            if (isset($paginatorConfig[PaginatorConfig::DEFAULT_PAGINATOR_NAME])) {
                $defaultPaginatorConfig = $paginatorConfig[PaginatorConfig::DEFAULT_PAGINATOR_NAME];
            }
            // vyvari defaultni strankovac pro jednoduche strankovani pres tridu
            $this->paginatorFactories[PaginatorConfig::DEFAULT_PAGINATOR_NAME] = new DefaultPaginatorFactory($this->domainClassName, $defaultPaginatorConfig);

            // projde ostatni strankovace
            foreach ($paginatorConfig as $paginatorName => $customPaginatorConfig) {
                if ($paginatorName != PaginatorConfig::DEFAULT_PAGINATOR_NAME) { // neni defaultni
                    $this->paginatorFactories[$paginatorName] = new CustomPaginatorFactory($paginatorName, $this->domainClassName, $customPaginatorConfig);
                }
            }
        } else {
            // vyvari defaultni strankovac pro jednoduche strankovani pres tridu
            $this->paginatorFactories[PaginatorConfig::DEFAULT_PAGINATOR_NAME] = new DefaultPaginatorFactory($this->domainClassName, []);
        }

        $this->log = Logger::getLogger($daoInterface);
    }

    /**
     * @inheritDoc
     */
    public function load($id)
    {
        return $this->dbConnection->load($this->domainClassName, $id);
    }

    /**
     * @inheritDoc
     */
    public function loadAll()
    {
        return $this->dbConnection->loadAll($this->domainClassName);
    }

    /**
     * @inheritDoc
     */
    public function find(array $params, array $orders = [])
    {
        return $this->dbConnection->find($this->domainClassName, $params, $orders);
    }

    /**
     * @inheritDoc
     */
    public function update($obj)
    {
        $this->dbConnection->update($obj);
    }

    /**
     * @inheritDoc
     */
    public function save($obj)
    {
        return $this->dbConnection->save($obj);
    }

    /**
     * @inheritDoc
     */
    public function delete($obj)
    {
        $this->dbConnection->delete($obj, $this->domainClassName);
    }

    /**
     * @inheritDoc
     */
    public function createDefaultPaginatorConfig($maxPageSize = PaginatorConfig::DEFAULT_MAX_PAGE_SIZE): PaginatorConfig
    {
        $paginatorName = PaginatorConfig::DEFAULT_PAGINATOR_NAME;
        $paginatorConfig = $this->createPaginatorConfig($paginatorName, $maxPageSize);

        return $paginatorConfig;
    }

    /**
     * @inheritDoc
     */
    public function createPaginatorConfig($paginatorName, $maxPageSize = PaginatorConfig::DEFAULT_MAX_PAGE_SIZE): PaginatorConfig
    {
        if (!is_int($maxPageSize)) { // @phpstan-ignore function.alreadyNarrowedType
            throw new Exception("parameter \$maxPageSize must be number");
        }
        if (!isset($this->paginatorFactories[$paginatorName])) {
            $message = 'Pro tridu ' . $this->domainClassName . ' neni definovany paginator ' . $paginatorName;
            $exception = new Exception($message);
            $this->log->error($message, ["exception" => $message]);
            throw $exception;
        }

        $paginatorFactory = $this->paginatorFactories[$paginatorName];
        $paginatorFactory->init($this->dbConnection);
        $paginatorConfig = $paginatorFactory->createPaginatorConfig($maxPageSize);
        return $paginatorConfig;
    }

    /**
     * @inheritDoc
     */
    public function paginate(PaginatorConfig $paginatorConfig): PageContent
    {
        $paginatorName = $paginatorConfig->getPaginatorName();
        if ($paginatorName == '') {
            $paginatorName = PaginatorConfig::DEFAULT_PAGINATOR_NAME;
        }

        if (!array_key_exists($paginatorName, $this->paginatorFactories)) {
            throw new Exception("Paginator " . $paginatorName . " is not defined for class " . $this->domainClassName);
        }

        // vrat factory co se ma starat o strankovani
        $paginatorFactory = $this->paginatorFactories[$paginatorName];
        $paginatorFactory->init($this->dbConnection);

        return $this->dbConnection->paginate($this->domainClassName, $paginatorFactory, $paginatorConfig);
    }

    /**
     *
     * @see \IZON\DB\GenericDao::findBy()
     */
    /*    public function findBy($propertyName, $value) {
            // TODO: Auto-generated method stub
        }
    */
    /**
     *
     * @see \IZON\DB\GenericDao::create()
     */
    public function create()
    {
        return new $this->domainClassName();
    }

    /**
     *
     * @see \IZON\DB\GenericDao::flush()
     */
    public function flush()
    {
        // TODO: Auto-generated method stub
    }

    /**
     * zapocne transakci
     */
    public function beginTransaction()
    {
        return $this->dbConnection->beginTransaction();
    }

    /**
     * commituje transakci
     */
    public function commit()
    {
        return $this->dbConnection->commit();
    }

    /**
     * rollbackuje transakci
     */
    public function rollBack()
    {
        return $this->dbConnection->rollBack();
    }

    /**
     *
     * @param string $methodName
     * @param mixed[] $args
     * @return mixed
     * @throws Exception
     */
    public function __call($methodName, array $args)
    {
        // TODO: kontrola jestli metoda existuje v rozhrani daa a zacina find
        if (StringUtils::startsWith($methodName, "update")) {
            return $this->dbConnection->executeUpdate($this->domainClassName, $this->queryFactory, $methodName, $args);
        } elseif (StringUtils::startsWith($methodName, "count")) {
            return $this->dbConnection->executeCount($this->domainClassName, $this->queryFactory, $methodName, $args);
        } elseif (StringUtils::startsWith($methodName, "customCount")) {
            if (!array_key_exists(1, $args) || $args[1] === null) {
                $args[1] = [];
            }
            return $this->dbConnection->executeCustomCount($this->domainClassName, $this->queryFactory, $methodName, $args[0], $args[1]);
        } elseif (StringUtils::startsWith($methodName, "delete")) {
            return $this->dbConnection->executeDelete($this->domainClassName, $this->queryFactory, $methodName, $args);
        } elseif (StringUtils::startsWith($methodName, "simple") || StringUtils::startsWith($methodName, "custom")) {
            if (!array_key_exists(1, $args) || $args[1] === null) {
                $args[1] = [];
            }
            return $this->dbConnection->executeCustomFind($this->domainClassName, $this->queryFactory, $methodName, $args[0], $args[1]);
        } else {
            // cokoliv, co nacita data z db
            return $this->dbConnection->executeFind($this->domainClassName, $this->queryFactory, $methodName, $args);
        }
    }
}
