<?php

namespace IZON\DB\Impl;

use Exception;
use IZON\DB\DBConnection;
use IZON\DB\Definitions\AbstractDefinition;
use IZON\DB\Definitions\CharDefinition;
use IZON\DB\Definitions\DateDefinition;
use IZON\DB\Definitions\DateTimeDefinition;
use IZON\DB\Definitions\FileDefinition;
use IZON\DB\Definitions\FloatDefinition;
use IZON\DB\Definitions\ImageDefinition;
use IZON\DB\Definitions\IntegerDefinition;
use IZON\DB\Definitions\TextDefinition;
use IZON\DB\Definitions\TimeDefinition;
use IZON\DB\Definitions\TransientDefinition;
use IZON\DB\Utils\ClassDescription;
use IZON\DB\Utils\PropertyDescriptions\BooleanDescription;
use IZON\DB\Utils\PropertyDescriptions\CharDescription;
use IZON\DB\Utils\PropertyDescriptions\DateDescription;
use IZON\DB\Utils\PropertyDescriptions\DateTimeDescription;
use IZON\DB\Utils\PropertyDescriptions\FileDescription;
use IZON\DB\Utils\PropertyDescriptions\FloatDescription;
use IZON\DB\Utils\PropertyDescriptions\ImageDescription;
use IZON\DB\Utils\PropertyDescriptions\IntegerDescription;
use IZON\DB\Utils\PropertyDescriptions\TextDescription;
use IZON\DB\Utils\PropertyDescriptions\TimeDescription;
use IZON\IO\File;
use IZON\IO\Image;
use IZON\Utils\Date;
use PDO;
use PDOStatement;

/**
 * Connector for PostgreSQL
 */
class DBConnectionPostgreSQLImpl extends DBConnectionCommon implements DBConnection
{
    /**
     * name of DB
     */
    public const DB_NAME = "PostgreSQL";

    //    private $dbName;

    /**
     * @var string pod jakym identifikatorem se ma pradet id vlozeneho zaznamu
     */
    protected string $insertNewIdIdentifier = "newid";


    public function __construct(
        $host,
        $dbName,
        $login,
        $password,
        $accessabeFileDir,
        $hiddenFilesDir,
        array $settings = ["codepage" => 'utf8']
    ) {
        $completeSettings = array_merge($this->getDefaultSettings(), $settings);
        $completeSettings['accessabeFileDir'] = $accessabeFileDir;
        $completeSettings['hiddenFilesDir'] = $hiddenFilesDir;

        $pdo = new PDO("pgsql:host=$host;dbname=$dbName;user=$login;password=$password");

        // nastavuje hlaseni chyb vyhozenim vyjimky
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        // nekonvertovat prazdne stringy a nully a naopak pri fetch neni jen pro oracle
        $pdo->setAttribute(PDO::ATTR_ORACLE_NULLS, PDO::NULL_NATURAL);

        parent::__construct($pdo, self::DB_NAME, $completeSettings);
    }

    public function getPropertyDescriptions(ClassDescription $classDescription, array $propertyDefinitions, array $propertyTypes)
    {
        $tableName = $classDescription->getTableName();
        $statement = $this->getPDO()->prepare("select * from information_schema.columns "
            . "where table_name = :tableName");

        //        $statement->bindValue(":dbName", $this->dbName);
        $statement->bindValue(":tableName", $tableName);
        $statement->execute();

        if ($statement->rowCount() == 0) { // kontrola jestli
            throw new Exception("Pro třídu " . $classDescription->getClassName() . " neexistuje tabulka " . $tableName);
        }

        $columnDescriptions = [];
        while ($row = $statement->fetch(PDO::FETCH_ASSOC)) {
            $columnDescriptions[$row["column_name"]] = $row;
        }
        $statement->closeCursor();
        $statement = null;

        $propertyDescriptions = [];

        foreach ($propertyDefinitions as $key => $value) {
            $propertyDefinition = $value;

            // transient se nemapuji
            if ($propertyDefinition instanceof TransientDefinition) {
                continue;
            }

            $columnName = null;
            if ($propertyDefinition !== null) {
                $columnName = $propertyDefinition->getColumnName();
            }
            if ($columnName == null) { // column doesn't have defined column name
                $columnName = $this->namingStrategy->propertyToColumnName($key);
            }

            // init to null
            $propertyDescription = null;
            if ($propertyDefinition !== null) { // initialize from PropertyDefinition
                $propertyDescription = $this->getPropertyDescriptionByProperityDefinition($classDescription, $propertyDefinition, $key, $columnName);
            }

            if (!isset($columnDescriptions[$columnName])) {
                throw new Exception('Pro property ' . $key . ' v tabulce ' . $tableName . ' neexistuje sploupec ' . $columnName);
            }
            $columnDescription = $columnDescriptions[$columnName];

            if ($propertyDescription == null) {
                $notNull = $columnDescription["is_nullable"] != "YES";
                $propertyDescription = $this->getPropertyDescriptionByDataType(
                    $classDescription,
                    $propertyTypes[$key],
                    $key,
                    $columnName,
                    $columnDescription["data_type"],
                    $notNull
                );
            }
            if ($propertyDescription == null) { // nastavit defaultni na char
                $propertyDescription = new CharDescription(
                    $classDescription,
                    $key,
                    $columnName,
                    mb_strtolower($key),
                    $columnDescription["is_nullable"] != "YES",
                    []
                );
            }

            // TODO: kontrola jestli odpovida informaci ziskanym z db

            $propertyDescriptions[$key] = $propertyDescription;
        }

        return $propertyDescriptions;
    }

    /**
     *
     * @param AbstractDefinition $propertyDefinition
     * @param string $propertyName
     * @param string $columnName
     * @return \IZON\DB\Utils\PropertyDescriptions\AbstractPropertyDescription|null
     */
    protected function getPropertyDescriptionByProperityDefinition(
        ClassDescription $classDescription,
        AbstractDefinition $propertyDefinition,
        $propertyName,
        $columnName
    ) {
        $columnReturnName = mb_strtolower($propertyName);
        $propertyDefinitionClass = get_class($propertyDefinition);

        switch ($propertyDefinitionClass) {
            case CharDefinition::class:
                $def = new CharDescription(
                    $classDescription,
                    $propertyName,
                    $columnName,
                    $columnReturnName,
                    $propertyDefinition->getNotNull(),
                    $propertyDefinition->getParameters()
                );
                return $def;
            case DateDefinition::class:
                // TODO: betters
                return new DateDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case TimeDefinition::class:
                return new TimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case DateTimeDefinition::class:
                return new DateTimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case IntegerDefinition::class:
                return new IntegerDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case FloatDefinition::class:
                return new FloatDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case CharDefinition::class:
                return new CharDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case TextDefinition::class:
                return new TextDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case FileDefinition::class:
                $params = $propertyDefinition->getParameters();
                $params["accessibleFileDir"] = $this->accessibleFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new FileDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $params);
            case ImageDefinition::class:
                $params = $propertyDefinition->getParameters();
                $params["accessibleFileDir"] = $this->accessibleFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new ImageDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $params);
        }

        if ($this->hasDescriptionFactoryForDefinition($propertyDefinitionClass)) {
            $params = $propertyDefinition->getParameters();
            // add info for files
            $params["accessibleFileDir"] = $this->accessibleFileDir;
            $params["hiddenFilesDir"] = $this->hiddenFilesDir;
            $params["missingFileFilePath"] = $this->missingFileFilePath;

            $descriptionFactory = $this->getDescriptionFactory($propertyDefinitionClass);
            return $descriptionFactory->createPropertyDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $params);
        }

        // none found
        return null;
    }

    protected function getPropertyDescriptionByDataType(
        ClassDescription $classDescription,
        $propertyType,
        $propertyName,
        $columnName,
        $columnDataType,
        $notNull
    ) {
        $columnReturnName = mb_strtolower($propertyName);

        //echo "$propertyType $propertyName $columnName $columnDataType\n";
        switch ($propertyType) {
            case Date::class: // pro date
                switch ($columnDataType) {
                    case "date":
                    case "year":
                        return new DateDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                    case "time":
                        return new TimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                    case "datetime":
                    case "timestamp":
                        return new DateTimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                    default:
                        throw new Exception("Data type $columnDataType not supported.");
                }
                // no break
            case File::class:
                $params = [];
                $params["accessibleFileDir"] = $this->accessibleFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new FileDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, $params);
            case Image::class:
                $params = [];
                $params["accessibleFileDir"] = $this->accessibleFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new ImageDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, $params);
            case "integer":
            case "int":
                return new IntegerDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
            case "float":
            case "double":
                return new FloatDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
            case "boolean":
            case "bool":
                return new BooleanDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
            case "string":
                switch ($columnDataType) {
                    case "text":
                        return new TextDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                    default:
                        return new CharDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                }
                // no break
            default:
                return null;
        }
    }

    protected function createInsertSQLString(ClassDescription $classDescription)
    {
        $columnNames = '';
        $properties = '';
        foreach ($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
            if ($key != $classDescription->getIdPropertyName()) { // updatuj vsechno krome id
                if ($properties != '') {
                    $columnNames .= ', ';
                    $properties .= ', ';
                }
                $columnNames .= $value;
                $properties .= ":" . $key;
            }
        }
        $sql = 'insert into ' . $classDescription->getTableName() . ' (' . $columnNames . ') ' .
            ' values (' . $properties . ') returning ' . $classDescription->getIdColumnName() . " as " . $this->insertNewIdIdentifier;

        return $sql;
    }

    /**
     * vraci vygenerovane id posledniho vlozene zaznamu
     * @param PDO $pdo pdo db pripojeni
     * @param PDOStatement $statement statement ktery byl vykonan
     * @return mixed returns last id from db
     */
    protected function getLastInsertedId(PDO $pdo, PDOStatement $statement)
    {
        $id = $statement->fetch(PDO::FETCH_ASSOC);
        return $id[$this->insertNewIdIdentifier];
    }
}
