<?php

namespace IZON\DB\Definitions;

use Exception;
use IZON\Logs\Logger;

/**
 * korenova definice, ktera slouzi jako zaklad pro vsechny definice
 */
abstract class AbstractDefinition
{
    /**
     * jak se ma nazyvat sloupecek pro property
     */
    protected ?string $columnName = null;

    protected bool $notNull = false;

    protected bool $readonly = false;

    /**
     * ostatni parametry, krome columnName a notNull,
     * ktere se extrahuji z definicniho pole
     * @var array
     */
    protected array $parameters = [];


    /**
     *
     * @var Logger
     */
    private $log;

    /**
     *
     * @param array $definitions
     * @throws Exception
     */
    public function __construct(array $definitions)
    {
        $this->columnName = $this->extractDefinition("columnName", $definitions);
        $this->notNull = $this->extractDefinition("notNull", $definitions, false, false);
        $this->readonly = $this->extractDefinition("readonly", $definitions, false, false);

        // zkontroluje parametry
        $params = [];
        $supported = $this->getSupportedParameters();
        foreach ($definitions as $key => $value) {
            if (in_array($key, $supported)) {
                $params[$key] = $value;
                unset($definitions[$key]);
            }
        }
        $this->parameters = $params;

        //        var_dump($params);

        if (!empty($definitions)) {
            //            var_dump($definitions);
            throw new Exception("Definitions " . implode(", ", $definitions) . " not supported");
        }
    }

    /**
     * extracts definition from array (returns value, unsets value in array), returns NULL if not
     * @param string $definitionName name of definition to extract
     * @param array $definitions definitions array
     * @param boolean $raiseExceprionIfNotPresent raise not present exception
     * @param mixed $default default value to extract
     * @return mixed|NULL extracted definition or $default if not present
     */
    protected function extractDefinition(
        $definitionName,
        array &$definitions, // pass by reference to unset work properly
        $raiseExceprionIfNotPresent = false,
        $default = null
    ) {
        $retval = $default;
        if (isset($definitions[$definitionName])) {
            $retval = $definitions[$definitionName];
            unset($definitions[$definitionName]);
        } elseif ($raiseExceprionIfNotPresent) {
            $e = new Exception("Definition $definitionName not set");
            $this->getLogger()->error("Definition $definitionName not set", ['exception' => $e]);
            throw $e;
        }
        return $retval;
    }

    protected function getLogger()
    {
        if ($this->log == null) {
            $this->log = Logger::getLogger(self::class);
        }
        return $this->log;
    }

    /**
     * vraci podporovane parametry pro danou definici
     */
    abstract protected function getSupportedParameters();

    public function getColumnName()
    {
        return $this->columnName;
    }

    public function getNotNull()
    {
        return $this->notNull;
    }

    public function isReadonly()
    {
        return $this->readonly;
    }

    /**
     * @return array array of deffinitions
     */
    public function getParameters()
    {
        return $this->parameters;
    }

    public function __toString()
    {
        // TODO: nahradit necim co vypise jak klice tak hodnoty
        return "[" . get_class($this) . ", " . implode(", ", array_map(function ($param) {
            return print_r($param, true);
        }, (array)$this)) . "]";
    }
}
