<?php

namespace IZON\DB;

use IZON\DB\Impl\DaoImpl;
use IZON\DB\Impl\DBConnectionCommon;
use IZON\DI\FactoryObject;
use IZON\Logs\Logger;
use Laminas\Code\Generator\ClassGenerator;
use Laminas\Code\Generator\MethodGenerator;
use Laminas\Code\Generator\ParameterGenerator;
use Laminas\Code\Reflection\ClassReflection;
use ReflectionClass;

/**
 * Slouzi k vytvoreni daa
 */
class DaoFactoryObject implements FactoryObject
{
    protected DBConnectionCommon $connection;

    /**
     * @var class-string root dao interface to
     */
    protected string $rootDaoInterface = Dao::class;

    /**
     * jmeno rozhrani, ktere ma toto dao implemetovat
     * @var class-string
     */
    protected string $interfaceName;

    protected ClassReflection $interfaceReflectionClass;

    /**
     *
     * @var class-string jaka trida slouzi jako zaklad Daa
     */
    protected string $daoImplClass = DaoImpl::class;

    /**
     * @var string[] zakladni interfacy daa od kterych se dedi
     */
    protected array $baseDaoInterfaces;


    /**
     * @var string v jakem adresari jsou ulozeny query factory relativne k adresari s dao
     */
    protected string $queryFactoriesDirName = "../DB";

    /**
     * @var string v jakem adresari jsou ulozeny query factory relativne k adresari s dao
     */
    protected string $paginatorsQueryFactoriesDirName = "../DB/Paginators";

    private Logger $log;


    public function __construct(DBConnectionCommon $connection, $interfaceName)
    {
        $this->connection = $connection;
        $this->interfaceName = $interfaceName;
        $this->interfaceReflectionClass = new ClassReflection($interfaceName); // class Foo of namespace A
        $this->baseDaoInterfaces = [Dao::class];

        $this->log = Logger::getLogger(self::class);
    }

    public function getObject()
    {
        $objectName = $this->interfaceName . "Impl";

        // trida jeste neni nactena do aplikace je ji treba vytvorit nebo nacist
        if (!class_exists($objectName)) {
            $daoImplCacheFileName = $this->connection->getDaoImplCacheDir() . "/" . str_replace('\\', '/', $objectName) . ".php";
            $daoFileLastModified = filemtime($this->interfaceReflectionClass->getFileName());

            if (
                $this->connection->getDaoImplCacheDir() != null // je definovan adresar kam cachovat daa
                && file_exists($daoImplCacheFileName) // uz existuje vygenerovane dao
                && $daoFileLastModified <= filemtime($daoImplCacheFileName)  // cache je novejsi nez soubor ve kterem byl vygenerovan
            ) {
                // vem dao z cache
                require $daoImplCacheFileName;
            } else {
                $class = new ClassGenerator(
                    $objectName,
                    $this->interfaceReflectionClass->getNamespaceName(),
                    null,
                    "\\" . $this->daoImplClass,
                    ["\\" . $this->interfaceName]
                );

                // funkce pro obaleni do stringu
                $quoteFunction = function ($element) {
                    return "'" . $element . "'";
                };

                // soubory, ve kterych se definuji daa
                $queryFactoryFiles = $this->getQueryFactoriesFiles($this->interfaceReflectionClass);
                $queryFactoryFilesArraySourceCode = array_map($quoteFunction, $queryFactoryFiles);
                $queryFactoryFilesArraySourceCode = "[" . implode(", ", $queryFactoryFilesArraySourceCode) . "]";
                //                var_dump($queryFactoryFilesArraySourceCode);

                // soubory, ve kterych se definuji daa
                $paginatorQueryFactoryFiles = $this->getPaginatorQueryFactoriesFiles($this->interfaceReflectionClass);
                $paginatorQueryFactoryFilesArraySourceCode = array_map($quoteFunction, $paginatorQueryFactoryFiles);
                $paginatorQueryFactoryFilesArraySourceCode = "[" . implode(", ", $paginatorQueryFactoryFilesArraySourceCode) . "]";

                // vytvori konstruktor pro predani parametru
                $method = new MethodGenerator(
                    "__construct",
                    [new ParameterGenerator("connection", "\IZON\DB\DBConnection"),
                        "interfaceName",
                        new ParameterGenerator("daoInterfaceReflectionClass", "\ReflectionClass")]
                );
                $method->setBody(
                    '$queryFactories = ' . $queryFactoryFilesArraySourceCode . ";\r\n" .
                    '$paginatorQueryFactories = ' . $paginatorQueryFactoryFilesArraySourceCode . ";\r\n" .
                    'return parent::__construct($connection, $interfaceName, $daoInterfaceReflectionClass, $queryFactories, $paginatorQueryFactories);'
                );
                $class->addMethodFromGenerator($method);

                // prochaci metody rozhrani daa
                foreach ($this->interfaceReflectionClass->getMethods() as $method) {
                    $declaringClass = $method->getDeclaringClass();
                    // jsou zakladni metody daa
                    if ($declaringClass->getName() == $this->rootDaoInterface) {
                        continue;
                    }

                    $methodGenerator = MethodGenerator::copyMethodSignature($method);
                    $methodGenerator->removeFlag(MethodGenerator::FLAG_INTERFACE);
                    foreach ($method->getParameters() as $parameter) {
                        $methodGenerator->setParameter(ParameterGenerator::fromReflection($parameter));
                    }

                    $methodGenerator->setBody('return parent::__call("' . $method->getName() . '", func_get_args());');
                    $class->addMethodFromGenerator($methodGenerator);
                }

                $generated = $class->generate();
                eval($generated);

                if ($this->connection->getDaoImplCacheDir() != null) { // je nastaven adresar pro cache
                    // zapsat do cache
                    $daoImplCacheDirName = dirname($daoImplCacheFileName);
                    if (!file_exists($daoImplCacheDirName)) { // neni soubor do ktereho se ma zapsat
                        if (!@mkdir($daoImplCacheDirName, 0777, true)) { // nevyhazovat warning pokud jiz adresar existuje
                            $this->log->warning("Nepodarilo se vytvorit adresar $daoImplCacheDirName pro cache daa $objectName");
                        }
                    }
                    if (!file_put_contents($daoImplCacheFileName, "<?php\r\n" . $generated, LOCK_EX)) {
                        $this->log->warning("Nepodarilo se zapsat cache $objectName daa do souboru $daoImplCacheFileName");
                    }
                    if (file_exists($daoImplCacheFileName)) {
                        chmod($daoImplCacheFileName, 0777);
                    }
                }
            }
        }

        $object = new $objectName($this->connection, $this->interfaceName, $this->interfaceReflectionClass);
        return $object;
    }

    /**
     * vrati cesty k souborum s poli kde se definuji query factory pro jednotlive dotazy
     * @param ReflectionClass $interfaceReflectionClass
     * @return array
     */
    protected function getQueryFactoriesFiles(ReflectionClass $interfaceReflectionClass)
    {
        $paths = [];
        if (in_array($interfaceReflectionClass->getName(), $this->baseDaoInterfaces)) {
            return $paths;
        }
        $interfaces = $interfaceReflectionClass->getInterfaces();
        foreach ($interfaces as $reflectionClass) {
            $paths = array_merge($paths, $this->getQueryFactoriesFiles($reflectionClass));
        }
        $daoInterfaceDefiningFileNamePath = dirname($interfaceReflectionClass->getFileName()) . "/" .
            $this->queryFactoriesDirName . "/" . $interfaceReflectionClass->getShortName() . ".php";
        if (file_exists($daoInterfaceDefiningFileNamePath)) {
            $paths = array_merge([$daoInterfaceDefiningFileNamePath], $paths);
        }
        return $paths;
    }

    /**
     * vrati cesty k souborum s poli kde se definuji query factory pro jednotlive strankovace
     * @param ReflectionClass $interfaceReflectionClass
     * @return string[]
     */
    protected function getPaginatorQueryFactoriesFiles(ReflectionClass $interfaceReflectionClass): array
    {
        $paths = [];
        if (in_array($interfaceReflectionClass->getName(), $this->baseDaoInterfaces)) {
            return $paths;
        }
        $interfaces = $interfaceReflectionClass->getInterfaces();
        foreach ($interfaces as $reflectionClass) {
            $paths = array_merge($paths, $this->getPaginatorQueryFactoriesFiles($reflectionClass));
        }
        $daoInterfaceDefiningFileNamePath = dirname($interfaceReflectionClass->getFileName()) . "/" .
            $this->paginatorsQueryFactoriesDirName . "/" . $interfaceReflectionClass->getShortName() . ".php";
        if (file_exists($daoInterfaceDefiningFileNamePath)) {
            $paths = array_merge([$daoInterfaceDefiningFileNamePath], $paths);
        }
        return $paths;
    }
}
