<?php

namespace IZON\DB\Utils\PropertyDescriptions;

use \PDO;
use \Exception;

use IZON\Logs\Logger;

use IZON\IO\File;

use IZON\DB\Utils\ClassDescription;
use function IZON\File\copy;

/**
 * Popis pukladani soubou
 */
class FileDescription extends AbstractPropertyDescription {

    /**
     * ma byt primo viditelny na webu
     * @var boolean
     */
    protected $webVisible = true;

    /**
     * v jakem podadresari maji byt ulozeny soubory pro tridu
     * @var string|null
     */
    protected $fileDir = NULL;

    /**
     * v jakem adresari se nachazeji soubory pro dane property
     * @var string|null
     */
    protected $storageDir = NULL;

    /**
     * @var string|null jak se ma jmenovat obrazek, co se nahradi pokud obrazk neexistuje v media
     */
    protected $missingFileFilePath = NULL;

    /**
     *
     * @var Logger
     */
    protected $log;

    /**
     *
     * @param string $propertyName
     * @param string $columnName
     * @param boolean $notNull
     * @param array $parameters
     */
    function __construct(ClassDescription $classDescription,
                            $propertyName, $columnName, $columnReturnName, $notNull, array $parameters = []) {
        $this->webVisible = $this->extractParameter("webVisible", $parameters, false, true);
        //
        $defaultFileDir = str_replace("_", "-", $classDescription->getTableName());
        $this->fileDir = $this->extractParameter("fileDir", $parameters, false, $defaultFileDir);

        if($this->webVisible) {
            $this->storageDir = $this->extractParameter("accessibleFileDir", $parameters, true);
        } else {
            $this->storageDir = $this->extractParameter("hiddenFilesDir", $parameters, true);
        }

        $this->missingFileFilePath = $this->extractParameter("missingFileFilePath", $parameters, true);

        $this->storageDir .= "/". $this->fileDir;
        $this->storageDir = \IZON\File\normalizeFileName($this->storageDir);

        parent::__construct($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, $parameters);

        $this->dbDataType = File::class;
        $this->pdoDataType = PDO::PARAM_STR;
        $this->ansiDataType = AbstractPropertyDescription::ANSI_DATA_TYPE_BIT_VARYING;

        $this->log = Logger::getLogger(__CLASS__);
    }

    public function parse($value, $object) {
        if ($value == NULL) {
            if ($this->getNotNull()) {
                throw new Exception("" . $this->getPropertyName() . " nesmi byt null");
            }
            return NULL;
        }

        $storageDir = $this->getStorageDir($object);
        $fileName = $storageDir . "/" . $value;
        if ( !file_exists($fileName) ) {
            $this->log->error("Db file " . $fileName . " isn't present on filesystem.");

            if( file_exists($this->missingFileFilePath) ) {
                $fileName = $this->missingFileFilePath;
            } else {
                $this->log->error("Rplacement form missing file " . $this->missingFileFilePath . " isn't present on filesystem.");
                return NULL;
            }

        }
        // create file
        $file = new File($fileName);

        return $file;
    }

    /**
     *
     * @param File $value
     * @return string|null
     * @throws Exception
     */
    public function serialize($value, $object) {
        if ($value == NULL) {
            if ($this->getNotNull()) {
                throw new Exception("" . $this->getPropertyName() . " nesmi byt null");
            }
            return NULL;
        }
        if (!($value instanceof File)) {
            throw new Exception("Hodnota musi byt null nebo " . File::class);
        }

        $serializedString =  $value->getFileName();
        if( !$this->isInFileStorage($value, $object) ) {
            $serializedString = File::getSafeFileName($serializedString); // vem bezpecne jmeno
        }

        return $serializedString;
    }

    function getPdoDataType($value) {
        if ($value == NULL)
            return PDO::PARAM_NULL;
        return $this->pdoDataType;
    }

    function isWebVisible() {
        return $this->webVisible;
    }

    /**
     * vrati informace do ktereho adresare souboru se ma nahrat
     * @param object $object
     * @return string where to store files
     */
    public function getStorageDir($object) {
        $propertyDir = str_replace("_", "-", $this->columnName);
        return $this->getObjectStorageDir($object) ."/". $propertyDir;
    }

    /**
     * vrati informace do ktereho adresare se maji nahrat soubory pro objekt
     * @param object $object
     */
    public function getObjectStorageDir($object) {
        $idPropertyName = $this->classDescription->getIdPropertyName();
        $objectId = $this->classDescription->getPropertyValue($object, $idPropertyName);

        return $this->storageDir ."/". $objectId;
    }

    public function postRecordSave($obj) {
        $file = $this->classDescription->getPropertyValue($obj, $this->getPropertyName());
        if( $file != NULL ) {
            $storageDir = $this->getStorageDir($obj);
            if( !file_exists($storageDir) ) { // vytvorit adresar pokud neexistuje
                mkdir($storageDir, 0777, true); // vytvari se rekurzivne
            }
            $safeFileName = File::getSafeFileName($file->getFileName());
            copy($file, $storageDir ."/". $safeFileName); // zkopirovat do file storage viditelnefo z web

            $movedFile = $this->parse($safeFileName, $obj);
            $this->classDescription->setPropertyValue($obj, $this->getPropertyName(), $movedFile);
        }
    }

    public function postRecordUpdate($obj) {
        /* @var $updateFile File */
        $updateFile = $this->classDescription->getPropertyValue($obj, $this->getPropertyName());
        $storageDir = $this->getStorageDir($obj);

        if(
            $updateFile != NULL
            && !$this->isInFileStorage($updateFile, $obj)
        ) { // neni ve file storage, je ho tam potreba presunout
            if( !file_exists($storageDir) ) { // vytvorit adresar pokud neexistuje
                mkdir($storageDir, 0777, true); // vytvari se rekurzivne TODO: asi resit prava nejak lepe
            }
            // vymaz puvodni obsah adresare s obrazky
            foreach(scandir($storageDir) as $file) {
                if ('.' === $file || '..' === $file) continue;
                else {
                    chmod("$storageDir/$file", 0777);
                    unlink("$storageDir/$file");
                }
            }
            $safeFileName = File::getSafeFileName($updateFile->getFileName());
            copy($updateFile, $storageDir ."/". $safeFileName); // zkopirovat do file storage viditelnefo z web

            // nacti do objektu ulozeny soubor
            $movedFile = $this->parse($safeFileName, $obj);
            $this->classDescription->setPropertyValue($obj, $this->getPropertyName(), $movedFile);
        } else if($updateFile === NULL) { // no file in property
            if( !file_exists($storageDir) ) {
                return;
            }
            // vymaz puvodni obsah adresare s obrazky
            foreach(scandir($storageDir) as $file) {
                if ('.' === $file || '..' === $file) continue;
                else {
                    chmod("$storageDir/$file", 0777);
                    unlink("$storageDir/$file");
                }
            }
        }
    }

    public function postRecordDelete($obj) {
        $storageDir = $this->getStorageDir($obj);

        if( file_exists($storageDir) ) {
            // vymaze obsah adresare
            foreach(scandir($storageDir) as $file) {
                if ('.' === $file || '..' === $file) {
                    continue;
                } else {
                    chmod("$storageDir/$file", 0777);
                    unlink("$storageDir/$file");
                }
            }
            // smazat adresar
            rmdir($storageDir);
        }

        // remove empty directory for object
        $objectStorageDir = $this->getObjectStorageDir($obj);
        if( file_exists($objectStorageDir)
            && $this->isDirEmpty($objectStorageDir) ) {
            rmdir($objectStorageDir);
        }
    }

    /**
     * returns true if $file is already stored in file storage
     * @param File $file
     * @return boolean
     */
    protected function isInFileStorage(File $file, $object) {
        // NOTE: probabbly test if file have same date and size
        $objectStorageDir = $this->getObjectStorageDir($object);
        return  \IZON\File\isInSubDir( \IZON\File\normalizeFileName($file->getFsPath()), \IZON\File\normalizePath($objectStorageDir));
    }

    protected function isDirEmpty(string $dir) {
        $handle = opendir($dir);
        while (false !== ($entry = readdir($handle))) {
            if ($entry != "." && $entry != "..") {
                closedir($handle);
                return false;
            }
        }
        closedir($handle);
        return true;
    }
}
