<?php

namespace IZON\DB\Impl;

use Exception;
use IZON\DB\DBConnection;
use IZON\DB\Definitions\AbstractDefinition;
use IZON\DB\Definitions\CharDefinition;
use IZON\DB\Definitions\DateDefinition;
use IZON\DB\Definitions\DateTimeDefinition;
use IZON\DB\Definitions\FileDefinition;
use IZON\DB\Definitions\FloatDefinition;
use IZON\DB\Definitions\ImageDefinition;
use IZON\DB\Definitions\IntegerDefinition;
use IZON\DB\Definitions\JsonDefinition;
use IZON\DB\Definitions\TextDefinition;
use IZON\DB\Definitions\TimeDefinition;
use IZON\DB\Definitions\TransientDefinition;
use IZON\DB\Utils\ClassDescription;
use IZON\DB\Utils\PropertyDescriptions\AbstractPropertyDescription;
use IZON\DB\Utils\PropertyDescriptions\BooleanDescription;
use IZON\DB\Utils\PropertyDescriptions\CharDescription;
use IZON\DB\Utils\PropertyDescriptions\DateDescription;
use IZON\DB\Utils\PropertyDescriptions\DateTimeDescription;
use IZON\DB\Utils\PropertyDescriptions\FileDescription;
use IZON\DB\Utils\PropertyDescriptions\FloatDescription;
use IZON\DB\Utils\PropertyDescriptions\ImageDescription;
use IZON\DB\Utils\PropertyDescriptions\IntegerDescription;
use IZON\DB\Utils\PropertyDescriptions\JsonDescription;
use IZON\DB\Utils\PropertyDescriptions\TextDescription;
use IZON\DB\Utils\PropertyDescriptions\TimeDescription;
use IZON\IO\File;
use IZON\IO\Image;
use IZON\Logs\Logger;
use IZON\Utils\Date;
use PDO;
use PDOStatement;


/**
 * MySQL db connector
 */
class DBConnectionMySQLImpl extends DBConnectionCommon implements DBConnection {

    /**
     * name of DB
     */
    const DB_NAME = "MySQL";

    private $dbName;

    /**
     * @var Logger
     */
    private $log;

    function __construct($host, $dbName, $login, $password,
                        $accessabeFileDir, $hiddenFilesDir,
                        array $settings = ["codepage" => 'utf8']) {
        $completeSettings = array_merge($this->getDefaultSettings(), $settings);
        $completeSettings['accessabeFileDir'] = $accessabeFileDir;
        $completeSettings['hiddenFilesDir'] = $hiddenFilesDir;

        // FIXME: kontrola, jestli se podarilo pripojit
        $pdo = new PDO("mysql:host=$host;dbname=$dbName;charset=". $completeSettings["codepage"], $login, $password);
        // neemuluj prepared statements pouzitelne od mysql 5.1.17
        // http://stackoverflow.com/questions/10113562/pdo-mysql-use-pdoattr-emulate-prepares-or-not
        $pdo->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);
        // nastavuje hlaseni chyb nayjimky vyjimky
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        // nekonvertovat prazdne stringy a nully a naopak pri fetch neni jen pro oracle
        $pdo->setAttribute(PDO::ATTR_ORACLE_NULLS, PDO::NULL_NATURAL);

        // disable strict group by if "onlyFullGroupBy" => false setting passed to connection
        if( array_key_exists('onlyFullGroupBy', $completeSettings) ) {
            if( $completeSettings['onlyFullGroupBy'] == false ) {
                $statemet = $pdo->query("SHOW VARIABLES LIKE 'sql_mode';");
                $modeString = $statemet->fetch(PDO::FETCH_ASSOC)['Value'];
                $statemet->closeCursor();
                $modeString = str_replace("ONLY_FULL_GROUP_BY", '', $modeString);
                $pdo->exec("set sql_mode='". $modeString ."';");
            }
        }

        $this->dbName = $dbName;

        $this->log = Logger::getLogger(self::class);

        parent::__construct($pdo, self::DB_NAME, $completeSettings);
    }


    protected function createInsertSQLString(ClassDescription $classDescription) {
        $columnNames = '';
        $properties = '';
        foreach($classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
            if( $key != $classDescription->getIdPropertyName()) { // updatuj vsechno krome id
                if($properties != '') {
                    $columnNames .= ', ';
                    $properties .= ', ';
                }
                $columnNames .= $value;
                $properties .= ":". $key;

            }
        }
        $sql = 'insert into '. $classDescription->getTableName() .' ('. $columnNames .') values ('. $properties .')';

        return $sql;
    }

    /**
     * vraci vygenerovane id posledniho vlozene zaznamu
     * @param PDO $pdo pdo db pripojeni
     * @param PDOStatement $statement statement ktery byl vykonan
     * @return mixed
     */
    protected function getLastInsertedId(PDO $pdo, PDOStatement $statement) {
        return $pdo->lastInsertId();
    }

    /**
     *
     * @param ClassDescription $classDescription
     * @param AbstractDefinition[] $propertyDefinitions
     * @param string[] $propertyTypes
     * @return AbstractPropertyDescription[]
     */
    public function getPropertyDescriptions(ClassDescription $classDescription, array $propertyDefinitions, array $propertyTypes) {
        $tableName = $classDescription->getTableName();

        $statement = $this->getPDO()->prepare("select * from information_schema.columns
                            where table_schema = :dbName and table_name = :tableName");
        $statement->bindValue(":dbName", $this->dbName);
        $statement->bindValue(":tableName", $tableName);
        $statement->execute();

        if($statement->rowCount() == 0) { // kontrola jestli
            throw new Exception("Pro třídu ". $classDescription->getClassName() ." neexistuje tabulka ". $tableName);
        }

        $columnDescriptions = [];
        while( $row = $statement->fetch(PDO::FETCH_ASSOC) ) {
            $columnDescriptions[$row["COLUMN_NAME"]] = $row;
        }

        $statement->closeCursor();
        $statement = null;

        $propertyDescriptions = [];
        foreach( $propertyDefinitions as $key => $value ) {
            $propertyDefinition = $value;

            // transient se nemapuji
            if($propertyDefinition instanceof TransientDefinition) {
                continue;
            }

            $columnName = NULL;
            if( $propertyDefinition != NULL ) {
                $columnName = $propertyDefinition->getColumnName();
            }
            if( $columnName == NULL ) { // column doesn't have defined column name
                $columnName = $this->namingStrategy->propertyToColumnName($key);
            }


            // init to null
            $propertyDescription = NULL;
            if( $propertyDefinition !== NULL ) { // initialize from PropertyDefinition
                $propertyDescription = $this->getPropertyDescriptionByProperityDefinition($classDescription, $propertyDefinition, $key, $columnName);
            }

            if( !isset($columnDescriptions[$columnName]) ) {
                throw new Exception('Pro property '.$key.' v tabulce '. $tableName .' neexistuje sploupec '.$columnName);
            }
            $columnDescription = $columnDescriptions[$columnName];

            if($propertyDescription == NULL) {
                $notNull = $columnDescription["IS_NULLABLE"] != "YES";
                $propertyDescription = $this->getPropertyDescriptionByDataType(
                    $classDescription,
                    $propertyTypes[$key],
                    $key,
                    $columnName,
                    $columnDescription["DATA_TYPE"],
                    $notNull
                );
            }
            if($propertyDescription == NULL) { // nastavit defaultni na char
                $propertyDescription = new CharDescription(
                    $classDescription,
                    $key,
                    $columnName,
                    $key,
                    $columnDescription["IS_NULLABLE"] != "YES",
                    []
                );
            }

            // TODO: kontrola jestli odpovida informaci ziskanym z db

            $propertyDescriptions[$key] = $propertyDescription;
        }

        return $propertyDescriptions;
    }

    /**
     *
     * @param AbstractDefinition $propertyDefinition
     * @param string $propertyName
     * @param string $columnName
     * @return AbstractPropertyDescription|null
     */
    protected function getPropertyDescriptionByProperityDefinition(
        ClassDescription   $classDescription,
        AbstractDefinition $propertyDefinition,
        string             $propertyName,
        string             $columnName
    ) {
        $columnReturnName = $propertyName;
        $propertyDefinitionClass = get_class($propertyDefinition);

        switch($propertyDefinitionClass) {
            case CharDefinition::class;
                return new CharDescription(
                    $classDescription,
                    $propertyName,
                    $columnName,
                    $columnReturnName,
                    $propertyDefinition->getNotNull(),
                    $propertyDefinition->getParameters()
                );
            case DateDefinition::class:
                // TODO: betters
                return new DateDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case TimeDefinition::class:
                return new TimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case DateTimeDefinition::class:
                return new DateTimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case IntegerDefinition::class:
                return new IntegerDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case FloatDefinition::class:
                return new FloatDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case TextDefinition::class:
                return new TextDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
            case FileDefinition::class:
                $params = $propertyDefinition->getParameters();
                $params["accessibleFileDir"] = $this->accessibleFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new FileDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $params);
            case ImageDefinition::class:
                $params = $propertyDefinition->getParameters();
                $params["accessibleFileDir"] = $this->accessibleFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new ImageDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $params);
             case JsonDefinition::class:
                return new JsonDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $propertyDefinition->getParameters());
        }

        if( $this->hasDescriptionFactoryForDefinition($propertyDefinitionClass) ) {
            $params = $propertyDefinition->getParameters();
            // add info for files
            $params["accessibleFileDir"] = $this->accessibleFileDir;
            $params["hiddenFilesDir"] = $this->hiddenFilesDir;
            $params["missingFileFilePath"] = $this->missingFileFilePath;

            $descriptionFactory = $this->getDescriptionFactory($propertyDefinitionClass);
            return $descriptionFactory->createPropertyDescription($classDescription, $propertyName, $columnName, $columnReturnName, $propertyDefinition->getNotNull(), $params);
        }

        // none found
        return null;
    }

    protected function getPropertyDescriptionByDataType(
        ClassDescription $classDescription,
        ?string $propertyType,
        string $propertyName,
        string $columnName,
        string $columnDataType,
        bool $notNull
    ) {
        $columnReturnName = $propertyName;
//        echo "$propertyType $propertyName $columnName\n";
        switch( $propertyType ) {
            case Date::class: // pro date
                switch( $columnDataType ) {
                    case "date":
                    case "year":
                        return new DateDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                    case "time":
                        return new TimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                    case "datetime":
                    case "timestamp":
                        return new DateTimeDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                    default:
                        throw new Exception("Data type $columnDataType not supported.");
                }
            case File::class:
                $params = [];
                $params["accessibleFileDir"] = $this->accessibleFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new FileDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, $params);
            case Image::class:
                $params = [];
                $params["accessibleFileDir"] = $this->accessibleFileDir;
                $params["hiddenFilesDir"] = $this->hiddenFilesDir;
                $params["missingFileFilePath"] = $this->missingFileFilePath;
                return new ImageDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, $params);
            case "integer":
            case "int":
                return new IntegerDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
            case "float":
            case "double":
                return new FloatDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
            case "boolean":
            case "bool":
                return new BooleanDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
            case "string":
                switch( $columnDataType ) {
                    case "text":
                        return new TextDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                    default:
                        return new CharDescription($classDescription, $propertyName, $columnName, $columnReturnName, $notNull, []);
                }
            default:
                return NULL;
        }
    }
}
