<?php

namespace IZON\DB\Paginator;

use \Exception;

/**
 * slouzi jako konfigurace strankovani
 * 
 * TODO: mozna vyresit inteligentneji nastavovani ostatnich parametru
 */
class PaginatorConfig {

    /**
     * jak se jmenuje defaultni strankovac pro danou tridu
     */
    const DEFAULT_PAGINATOR_NAME = "_default";


    /**
     * defaultni velikost stranky
     */
    const DEFAULT_MAX_PAGE_SIZE = 100;

    /**
     * @var string|null jak se jmenuje strankovac, ktery ma tuto stranku obslouzit
     */
    protected $paginatorName = NULL;

    /**
     * pocet zaznamu na stranku
     */
    protected $maxPageSize;

    /**
     * @var integer index of first result 
     */
    protected $firstResult = 0;

    /**
     * @var array[]|null jaka pole jsou definovana pro tento strankovac
     */
    protected $definedFields = NULL;

    /**
     *
     * @var string[]|null podle kterych poli je mozno radit 
     */
    protected $definedOrders = NULL;

    /**
     * @var array parametry strankovace  
     */
    protected $parameters = [];


    /**
     * jednotlive sloupce podle kterych se ma radit ['propertyPodleKtereSeMaRadit', asc/desc]
     * @var array pole razeni
     */
    protected $orders = [];

    /**
     * 
     * @param string $paginatorName 
     * @param array $definedFields pole, ktera se mohou nastavit z formulare nebo jinym zpusobem z requestu
     * @param integer $maxPageSize
     */
    public function __construct(
        $paginatorName, 
        array $definedFields = [],
        $definedOrders = [],
        $maxPageSize = self::DEFAULT_MAX_PAGE_SIZE
    ) {
        $this->paginatorName = $paginatorName;
        $this->definedFields = $definedFields;
        $this->definedOrders = $definedOrders;
        $this->maxPageSize = $maxPageSize;
        
        if( !is_array($definedOrders) ) { // hack pro spetnou kompatibilitu
            $this->maxPageSize = $definedOrders;
            $this->definedOrders = [];
        }
    }

    function getPaginatorName() {
        return $this->paginatorName;
    }

    public function getMaxPageSize() {
        return $this->maxPageSize;
    }

    public function setMaxPageSize($maxPageSize) {
        $this->maxPageSize = $maxPageSize;
    }

    public function getFirstResult() {
        return $this->firstResult;
    }

    public function setFirstResult($firstResult) {
        $this->firstResult = $firstResult;
    }

    function getDefinedFields() {
        return $this->definedFields;
    }

    
    function getOrders() {
        return $this->orders;
    }
    
    function setOrders($orders) {
        // TODO: test ze je pole poli ve spravnem tvaru
        return $this->orders = $orders;
    }
    
    /**
     * 
     * @param string $propertyName na jake propery
     * @param boolean $asc jestli ma byt asc
     */
    function addOrder($propertyName, $asc = true) {
        $this->orders[] = [$propertyName, $asc];
    }
    
    function clearOrders() {
        $this->orders = [];
    }
    
    function isOrderable($propertyName) {
        return in_array($propertyName, $this->definedOrders);
    }
    
    /**
     * 
     * @param string $propertyName
     * @return boolean
     */
    function hasParameter($propertyName) {
        return isset($this->parameters[$propertyName]);
    }
            
    function putParameter($paramName, $paramValue) {
        if( !$this->__isset($paramName) ) {
            throw new Exception("Property $paramName not defined");
        }
        $this->parameters[$paramName] = $paramValue;
    }
    
    function getParameter($paramName) {
        if (!$this->__isset($paramName)) {
            throw new Exception("Property $paramName not defined");
        }
        if (!array_key_exists($paramName, $this->parameters)) {
            return null;
        }
        return $this->parameters[$paramName];
    }
    
    function removeParameter($paramName) {
        if (!$this->__isset($paramName)) {
            throw new Exception("Property $paramName not defined");
        }
        unset($this->parameters[$paramName]);
    }
    
    /**
     * @return array pole se vsemi nastavenymi parametry
     */
    function getParameters() {
        $params = [];
        foreach($this->parameters as $propertyName => $value) {
            if( $this->__isset($propertyName)
                && $value !== NULL ) {
                $params[$propertyName] = $value;
            }
        }
        
//        var_dump($params);
        return $params;
    }
            
    function __isset($propertyName) {
        foreach($this->definedFields as $definedFieldName) {
            $result = preg_match("#^". $definedFieldName ."$#", $propertyName);
            if( $result === FALSE ) {
                throw new Exception("Can't match deffined field ". $definedFieldName ." pattern");
            } else if($result) {
                return true;
            }
        }
        return false;
    }

    function __get($propertyName) {
        if (!$this->__isset($propertyName)) {
            throw new Exception("Property $propertyName not defined");
        }
        return $this->getParameter($propertyName);
    }

    function __set($propertyName, $value) {
//        var_dump($propertyName);
        if (!$this->__isset($propertyName) ) {
            throw new Exception("Property $propertyName not defined");
        }
        return $this->parameters[$propertyName] = $value;
    }

    /**
     * aby se dalo volat get[$name]
     * @param string $name
     * @param array $arguments
     */
    public function __call($name, array $arguments) {
        if (\IZON\String\startsWith($name, "get")) { // obsluhuji se jen get metody
            $propertyName = mb_substr($name, 3); // odstrani get
            $propertyName = lcfirst($propertyName); // prvni znak preved na mala pismena

            if (!$this->__isset($propertyName)) { // talove pole není definovano
                throw new Exception("Property $propertyName not defined");
            }
            return $this->getParameter($propertyName);
        } else if (\IZON\String\startsWith($name, "set")) { // obsluhuji se jen get metody
            $propertyName = mb_substr($name, 3); // odstrani set
            $propertyName = lcfirst($propertyName); // prvni znak preved na mala pismena

            if ($this->__isset($propertyName)) { // talove pole je definovano
                // ziskal argument pro nastaveni
                $firstKey = reset($arguments);
                // nastav hodnotu
                $this->parameters[$propertyName] = $firstKey;
            } else {
                throw new Exception("Property $propertyName not defined");
            }
        }
    }

    function getDefinedOrders() {
        return $this->definedOrders;
    }
}
