<?php
namespace IZON\DB\Paginator;

use Exception;

use \PDO;
use \PDOStatement;


use IZON\DB\Impl\DBConnectionCommon;
use IZON\DB\QueryHelpers;
use IZON\DB\Paginator\PageContent;

use IZON\DB\Utils\PropertyDescriptions\AbstractPropertyDescription;

/**
 * vrati stranku
 */
class DefaultPaginatorFactory implements PaginatorFactory {

    /**
     * @var string class of object this paginator is for
     */
    protected $class;

    /**
     * @var array property, podle kterych je mozne vyhledavat
     */
    protected $propertes = [];
    
    /**
     * @var array|null podle kterych property je mozne radit  
     */
    protected $orders = NULL;

    /**
     * @var array jake je defaultni razeni 
     */
    protected $defaultOrder = ["id", false];
    
    /**
     * @var array|null konfugurace ziskana z konfiguracniho souboru 
     */
    protected $config = null;
            
    function __construct($class, array $config = []) {
        $this->class = $class;
        $this->config = $config;
        
//        $classDescription = $connection->getClassDesctiption($this->class);
        
//        $columnTranslations = $classDescription->getPropertiesToColumnsTranslation();
//        foreach($columnTranslations as $columnTranslation) {
//            
//        }
    }
    
    /**
     * provede inicializaci sptankovace
     * @param DBConnectionCommon $connection
     */
    public function init(DBConnectionCommon $connection) {
        if( is_null($this->config) )  {// konfig uz byl parsovan, neni delat znova
            return;
        }
        $config = $this->config;
        
        $properties = [];
        $orders = [];
        $classDescription = $connection->getClassDesctiption($this->class);
        foreach($classDescription->getPropertiesToColumnsTranslation() as $propertyName => $colName) {
            $propDescription = $classDescription->getPropertyDescription($propertyName);
            $dbDataType = $propDescription->getAnsiDataType();
            
            if( isset($config["excludedOrders"]) ) {
                // podle ceho lze radit
                if( !in_array($propertyName, $config["excludedOrders"] ) // neni vyhozeno z razeni
                    && !in_array($dbDataType, [AbstractPropertyDescription::ANSI_DATA_TYPE_BIT_VARYING]) ) { // je nejaky bitovy blob podle ktereho neni mozne radit
                    $orders[] = $propertyName; 
                }
            }
            // property se ma ignorocat pro vyhledavani
            if( isset($config["excludedProperties"])
                && in_array($propertyName, $config["excludedProperties"]) ) { 
                continue;
            }
            
            $propParams = [];
            if( in_array($dbDataType, [AbstractPropertyDescription::ANSI_DATA_TYPE_BIT_VARYING]) ) { // je nejaky bitovy blob, ve kterem neni mozno vyhledavat
                continue;
            }
            // nastavuje defaultni hodnoty
            if( in_array($dbDataType, [AbstractPropertyDescription::ANSI_DATA_TYPE_CHAR_VARYING]) ) { // je nejaky retezcovy datovy typ
                $propParams["operation"] = "like";
                $propParams["valuePrefix"] = "%";
                $propParams["valueSufix"] = "%";
                $propParams["emptyCanBeNull"] = true;
            } else {
                $propParams["operation"] = "=";
                $propParams["valuePrefix"] = "";
            }
            
            // parsuje konfig pole
             if( isset($config["properyDefinitions"][$propertyName]) ) {
                $propDef = $config["properyDefinitions"][$propertyName];
                if( isset($propDef["operation"]) ) {
                    $propParams["operation"] = $propDef["operation"];
                }
                if( isset($propDef["valuePrefix"]) ) {
                    $propParams["valuePrefix"] = $propDef["valuePrefix"];
                }
                if( isset($propDef["valueSufix"]) ) {
                    $propParams["valueSufix"] = $propDef["valueSufix"];
                }
                if( isset($propDef["emptyCanBeNull"]) ) {
                    $propParams["emptyCanBeNull"] = $propDef["emptyCanBeNull"];
                }
            }
            
            $properties[$propertyName] = $propParams;
        }
        
        $this->orders = $orders;
        $this->propertes = $properties;
        
        if( isset($config["defaultOrder"]) ) {
            // TODO: kontrola ze ma spravny tvar
            $this->defaultOrder = $config["defaultOrder"];
        }
        
        $this->config = NULL;
    }

    /**
     * vytvori paginator config
     */
    public function createPaginatorConfig($maxPageSize) {
        $config = new PaginatorConfig(PaginatorConfig::DEFAULT_PAGINATOR_NAME, array_keys($this->propertes), $this->orders, $maxPageSize);
        $config->addOrder($this->defaultOrder[0], $this->defaultOrder[1]);
        return $config;
    }

    /**
     * 
     * @param QueryHelpers $queryHelpers
     * @param PaginatorConfig $config
     * @return string
     */
    public function page(QueryHelpers $queryHelpers, PaginatorConfig $config) {
        $query = "select " . $queryHelpers->getClassProperties("t")
                . " from " . $queryHelpers->getTableName("t");
        
        $queryParams = $this->parseWhereParameters($queryHelpers, $config);
        if( !\IZON\Arrays\isEmpty($queryParams) ) {
             $query .= " where ". implode(" and ", $queryParams);
        }
        
        $orderByPars = [];
        foreach( $config->getOrders() as $order) {
            $orderByPart = " ". $queryHelpers->getCoumnName($order[0], "t") ." ";
            if( $order[1] ) {
                $orderByPart .= " asc";
            } else {
                $orderByPart .= " desc";
            }
            $orderByPars[] = $orderByPart;
        }
        
        if( !\IZON\Arrays\isEmpty($orderByPars) ) {
            $query .= " order by ". implode(", ", $orderByPars);
        }
        
        $query .= " limit :maxResults offset :firstResult";
        
        return $query;
    }
    
    /**
     * 
     * @param QueryHelpers $queryHelpers
     * @param \IZON\DB\Paginator\PaginatorConfig $config
     * @return string
     */
    public function count(QueryHelpers $queryHelpers, PaginatorConfig $config) {
        $query = "select count(*) as ". $queryHelpers->getCountAlias() ." "
                . " from " . $queryHelpers->getTableName("t");

        $queryParams = $this->parseWhereParameters($queryHelpers, $config);
        if( !\IZON\Arrays\isEmpty($queryParams) ) {
             $query .= " where ". implode(" and ", $queryParams);
        }
      
//        echo $query;
        return $query;
    }
    
    /**
     * parsuje parametry where casti dotazu
     * @param QueryHelpers $queryHelpers
     * @param PaginatorConfig $config
     * @throws Exception
     */
    protected function parseWhereParameters(QueryHelpers $queryHelpers, PaginatorConfig $config) {
        $propertes = $this->propertes;
        $queryParams = [];
        foreach($propertes as $propName => $property) {
            if( !$config->hasParameter($propName) ) { // takovy parametr neby nastaven
                continue;
            }
            $columName = $queryHelpers->getCoumnName($propName, "t");
            $queryParam = $columName ." ". $property["operation"] ." ";
            if( $property["valuePrefix"] != ""
                && $property["valueSufix"] != "" ) {
                $queryParam .= "concat('". $property["valuePrefix"] ."', ". $queryHelpers->getParameterName($propName) .", '". $property["valueSufix"] ."')";
            } else if( $property["valuePrefix"] != "" ) {
                $queryParam .= "concat('". $property["valuePrefix"] ."', ". $queryHelpers->getParameterName($propName) .")";
            } else if( $property["valueSufix"] != "" ) {
                $queryParam .= "concat(". $queryHelpers->getParameterName($propName) .", '". $property["valueSufix"] ."')";
            } else {
                $queryParam .= $queryHelpers->getParameterName($propName);
            }
            // 
            if( $property["emptyCanBeNull"] ) {
                $queryParam = " (". $queryParam ." or (". $queryHelpers->getParameterName($propName) ." = '' and ". $columName ." is null)) ";
            }
            
            $queryParams[] = $queryParam;
        }
        
        return $queryParams;
    }

}
