<?php
namespace IZON\DB\Impl;

use ReflectionClass;

use Exception;

use IZON\DB\Dao;
use IZON\DB\Paginator\PaginatorConfig;
use IZON\DB\DBConnection;
use IZON\Logs\Logger;

use IZON\DB\QueryFactory;
use IZON\DB\QueryHelpers;

use IZON\DB\Paginator\DefaultPaginatorFactory;
use IZON\DB\Paginator\CustomPaginatorFactory;

class DaoImpl implements Dao {

    /**
     * @var DBConnectionCommon 
     */
    protected $dbConnection;
    
    /**
     * @var ReflectionClass reflection class
     */
    protected $daoInterfaceReflectionClass;
    
    /**
     * domenova trida, ktera se ma vracet
     * @var string 
     */
    protected $domainClassName;
    
    /**
     * query factory, ktera slouzi v vhytvareni fotazu pro jednotlive metody daa
     * @var QueryFactory|null
     */
    protected $queryFactory = NULL;

    /**
     * strankovace, ktere se mohou pouzit pro $domainClassName
     * @var array 
     */
    protected $paginatorFactories = [];
    
    /**
     * @var Logger 
     */
    private $log;
    
    
    function __construct(DBConnectionCommon $dbConnection, $daoInterface, 
                            ReflectionClass $daoInterfaceReflectionClass, 
                            $queryFactoriesFiles, $paginatorQueryFactoriesFiles) {
        $this->dbConnection = $dbConnection;
        $this->daoInterfaceReflectionClass = $daoInterfaceReflectionClass;
        $this->domainClassName = $daoInterface::DOMAIN_CLASS;
        
//        $definingFileDir = dirname($daoInterfaceReflectionClass->getFileName());
//        $queryFactoryConfig = dirname($definingFileDir) ."/DB/". $daoInterfaceReflectionClass->getShortName() .".php";
        
        $queryConfig = [];
        foreach($queryFactoriesFiles as $fileName) {
            $queryConfig = array_merge($queryConfig, require $fileName);
        }
        if( !\IZON\Arrays\isEmpty($queryConfig) ) {
            $this->queryFactory = $queryConfig;
        } else {
            // nacte potrebnou query factory podle domain class
           $queryFactoryName = str_replace("\\Domain\\", "\\DB\\", $this->domainClassName) ."QueryFactory". $this->dbConnection->getDbType();
           if( class_exists($queryFactoryName) ) {
               $this->queryFactory = new $queryFactoryName();
           } else {
               $queryFactoryName = str_replace("\\Domain\\", "\\DB\\", $this->domainClassName) ."QueryFactory";
//               var_dump($queryFactoryName);
               if( class_exists($queryFactoryName) ) {
                   $this->queryFactory = new $queryFactoryName();
//                   var_dump(new $queryFactoryName());
               }
           }
        }
        
        // zpracuje paginator factory
        $paginatorConfig = [];
        foreach($paginatorQueryFactoriesFiles as $fileName) {
//            var_dump($fileName);
            $paginatorConfig = array_merge($paginatorConfig, require $fileName);
        }
        if( !\IZON\Arrays\isEmpty($paginatorConfig) ) { // mame konfig
            $defaultPaginatorConfig = [];
            if( isset($paginatorConfig[PaginatorConfig::DEFAULT_PAGINATOR_NAME]) ) {
                $defaultPaginatorConfig = $paginatorConfig[PaginatorConfig::DEFAULT_PAGINATOR_NAME];
            }
            // vyvari defaultni strankovac pro jednoduche strankovani pres tridu
            $this->paginatorFactories[PaginatorConfig::DEFAULT_PAGINATOR_NAME] = new DefaultPaginatorFactory($this->domainClassName, $defaultPaginatorConfig);
            
            // projde ostatni strankovace
            foreach($paginatorConfig as $paginatorName => $customPaginatorConfig) {
                if( $paginatorName != PaginatorConfig::DEFAULT_PAGINATOR_NAME ) { // neni defaultni
                    $this->paginatorFactories[$paginatorName] = new CustomPaginatorFactory($paginatorName, $this->domainClassName, $customPaginatorConfig);
                }
            }
        } else {
            // vyvari defaultni strankovac pro jednoduche strankovani pres tridu
            $this->paginatorFactories[PaginatorConfig::DEFAULT_PAGINATOR_NAME] = new DefaultPaginatorFactory($this->domainClassName, []);
        }
        
        $this->log = Logger::getLogger($daoInterface);
    }
    
    /**
     *
     * @see \IZON\DB\GenericDao::load()
     */
    public function load($id) {
        return $this->dbConnection->load($this->domainClassName, $id);
    }

    /**
     *
     * @see \IZON\DB\GenericDao::loadAll()
     */
    public function loadAll() {
        return $this->dbConnection->loadAll($this->domainClassName);
    }
    
    public function find(array $params, array $orders = []) {
        return $this->dbConnection->find($this->domainClassName, $params, $orders);
    }
    
    /**
     *
     * @see \IZON\DB\GenericDao::update()
     */
    public function update($obj) {
        $this->dbConnection->update($obj);
    }

    /**
     *
     * @see \IZON\DB\GenericDao::save()
     */
    public function save($obj) {
        return $this->dbConnection->save($obj);
    }

    /**
     *
     * @see \IZON\DB\GenericDao::delete()
     */
    public function delete($obj) {
        $this->dbConnection->delete($obj, $this->domainClassName);
    }

    public function createPaginatorConfig($paginatorName, $maxPageSize = PaginatorConfig::DEFAULT_MAX_PAGE_SIZE) {
        if( !is_int($maxPageSize) ) {
            throw new Exception("parametr \$maxPageSize musi byt cislo");
        }
        if( !isset($this->paginatorFactories[$paginatorName]) ) {
            $message = 'Pro tridu '. $this->domainClassName .' neni definovany paginator '. $paginatorName;
            $exception = new \Exception($message);
            $this->log->error($message, ["exception" => $message]);
            throw $exception;
        }
        
        $paginatorFactory = $this->paginatorFactories[$paginatorName];
        $paginatorFactory->init($this->dbConnection);
        $paginatorConfig = $paginatorFactory->createPaginatorConfig($maxPageSize);
        return $paginatorConfig;
    }
    
    public function createDefaultPaginatorConfig($maxPageSize = PaginatorConfig::DEFAULT_MAX_PAGE_SIZE) {
        $paginatorName = PaginatorConfig::DEFAULT_PAGINATOR_NAME;
        $paginatorConfig = $this->createPaginatorConfig($paginatorName, $maxPageSize);
        
        return $paginatorConfig;
    }
    
    /**
     *
     * @see \IZON\DB\GenericDao::paginate()
     */
    public function paginate(PaginatorConfig $config) {
        $paginatorName = $config->getPaginatorName();
        if( $paginatorName == '') {
            $paginatorName = PaginatorConfig::DEFAULT_PAGINATOR_NAME;
        }

        if( !array_key_exists($paginatorName, $this->paginatorFactories) ) {
            throw new Exception("Paginator ". $paginatorName ." is not defined for class ". $this->domainClassName);
        }
        
        // vrat factory co se ma starat o strankovani
        $paginatorFactory = $this->paginatorFactories[$paginatorName];
        $paginatorFactory->init($this->dbConnection);
        
        return $this->dbConnection->paginate($this->domainClassName, $paginatorFactory, $config);
    }

    /**
     *
     * @see \IZON\DB\GenericDao::findBy()
     */
/*    public function findBy($propertyName, $value) {
        // TODO: Auto-generated method stub
    }
*/
    /**
     *
     * @see \IZON\DB\GenericDao::create()
     */
    public function create() {
        return new $this->domainClassName;
    }

    /**
     *
     * @see \IZON\DB\GenericDao::flush()
     */
    public function flush() {
        // TODO: Auto-generated method stub
    }    
    
    /**
     * zapocne transakci
     */
    public function beginTransaction() {
        return $this->dbConnection->beginTransaction();
    }
    
    /**
     * commituje transakci
     */
    public function commit() {
        return $this->dbConnection->commit();
    }
    
    /**
     * rollbackuje transakci
     */
    public function rollBack() {
        return $this->dbConnection->rollBack();
    }
    
    /**
     * 
     * @param string $methodName
     * @param mixed[] $args
     * @return mixed
     */
    function __call($methodName, array $args) {
        // TODO: kontrola jestli metoda existuje v rozhrani daa a zacina find
        if(\IZON\String\startsWith($methodName, "update")) {
            return $this->dbConnection->executeUpdate($this->domainClassName, $this->queryFactory, $methodName, $args);
        } elseif (\IZON\String\startsWith($methodName, "count")) {
            return $this->dbConnection->executeCount($this->domainClassName, $this->queryFactory, $methodName, $args);
        } elseif (\IZON\String\startsWith($methodName, "customCount")) {
            if( $args[1] == NULL ) {
                $args[1] = [];
            }
            $ret = $this->dbConnection->executeCustomCount($this->domainClassName, $this->queryFactory, $methodName, $args[0], $args[1]);
            return $ret;
        } elseif (\IZON\String\startsWith($methodName, "delete")) {
            return $this->dbConnection->executeDelete($this->domainClassName, $this->queryFactory, $methodName, $args);
        } elseif (\IZON\String\startsWith($methodName, "simple") || \IZON\String\startsWith($methodName, "custom")) {
            if( $args[1] == NULL ) {
                $args[1] = [];
            }
            $ret = $this->dbConnection->executeCustomFind($this->domainClassName, $this->queryFactory, $methodName, $args[0], $args[1]);
            return $ret;
        } else {
            // cokoliv, co nacita data z db
            $ret = $this->dbConnection->executeFind($this->domainClassName, $this->queryFactory, $methodName, $args);
            return $ret;
        }
    }

}