<?php

namespace IZON\DB;

use Exception;

use IZON\DB\Utils\ClassDescription;
use IZON\DB\Impl\DBConnectionCommon;

/**
 * Slouzi k ulehceni prace pri sestavovani dotazu ve QueryFactory
 */
class QueryHelpers {
    
    /**
     * popis tridy pro kterou se QueryHelpers vytvari
     * @var ClassDescription  
     */
    protected $classDescription;
    
     /**
     * popisy vsech trid pro 
     * @var DBConnectionCommon 
     */
    protected $dBConnection;

    /**
     * @var array hodnoty parametru predavanych dotazu
     */
    protected $parameters;
    
    /**
     * @var array informace o tom ktere parametry byly pouzity vicekrat
     */
    protected $repetedParameters;
            
    /**
     * @var array ktere parametry byly expandovany na in pole 
     */
    protected $inExpandedParameters;
    
    /**
     * @var string pod jaky aliasem je ulozen pocet zaznamu
     */
    protected $countAlias = "count";
            
    function __construct(ClassDescription $classDescription, 
                        DBConnectionCommon $dBConnection,
                        array $parameters) {
        $this->classDescription = $classDescription;
        $this->dBConnection = $dBConnection;
        $this->parameters = $parameters;
        
        $this->repetedParameters = [];
        $this->inExpandedParameters = [];
    }

    /**
     * vrati cast dotazu po select ve tvaru "$alias.conum_name1 as propertyName1, $alias.conum_name2 as propertyName2, ... " 
     * @param string $alias
     * @return string
     */
    public function getClassProperties($alias = NULL) {
        $properties = '';
        foreach($this->classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
            $propertyDescription = $this->classDescription->getPropertyDescription($key);
            if($properties != '') {
                $properties .= ', ';
            }
            $properties .=  ($alias === NULL ? "" : "$alias.") . $value .' as '. $propertyDescription->getColumnReturnName();
        }
        return $properties;
    }
    
    /**
     * vrati cast dotazu po select ve tvaru "$alias.conum_name1 as propertyName1, $alias.conum_name2 as propertyName2, ... " 
     * 
     * !!! COMPATIBLE ONLY WITH MySQL !!!
     * 
     * @param string $alias
     * @return string
     */
    public function getQuotedClassProperties($alias = NULL) {
        $properties = [];
        foreach($this->classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
            $propertyDescription = $this->classDescription->getPropertyDescription($key);
            $properties[] =  ($alias === NULL ? '' : $alias.'.') .'`'.$value .'` as `'. $propertyDescription->getColumnReturnName() .'`';
        }
        return implode(',', $properties);
    }
    
    /**
     * vraci jmeno tabulku pripadne s aliasem
     * @param string jaky alias se ma pouzit
     * @return string nazev tabulky
     */
    public function getTableName($alias = NULL) {
        $tableName = $this->classDescription->getTableName();
        if( $alias !== NULL ) {
            $tableName = $tableName ." ". $alias;
        }
        return $tableName;
    }
    
    /**
     * @param class Jmemo tridy pro kterou se ma bratit
     * @return string nazev tabulky
     */
    public function getClassTableName($className, $alias = NULL) {
        $classDescription = $this->dBConnection->getClassDesctiption($className);
        $tableName = $classDescription->getTableName();
        if( $alias !== NULL ) {
            $tableName = $tableName ." ". $alias;
        }
        return $tableName;
    }
    
    /**
     * vrati nazev sloupce ve tvaru $alias.column_name kde column_name odpovida $propertyName ve tride
     * @param string $propertyName jmeno property, pro kteru ze ma najit 
     * @param string $alias
     */
    public function getColumnName($propertyName, $alias = NULL) {
        $columnName = $this->classDescription->getColumnName($propertyName);
        if( $alias !== NULL ) {
            $columnName = $alias .".". $columnName;
        }
        return $columnName;
    }
    
    /**
     * @deprecated use getColumnName
     * vrati nazev sloupce ve tvaru $alias.column_name kde column_name odpovida $propertyName ve tride
     * @param string $propertyName jmeno property, pro kteru ze ma najit 
     * @param string $alias
     */
    public function getCoumnName($propertyName, $alias = NULL) {
        return $this->getColumnName($propertyName, $alias);
    }

    /**
     * vrati nazev sloupce ve tvaru $alias.column_name kde column_name odpovida $propertyName ve tride
     * @param string $className
     * @param string $propertyName
     * @param string $alias
     */
    public function getClassColumnName($className, $propertyName, $alias = NULL) {
        $classDescription = $this->dBConnection->getClassDesctiption($className);
        $columnName = $classDescription->getColumnName($propertyName);
        if( $alias !== NULL ) {
            $columnName = $alias .".". $columnName;
        }
        return $columnName;
    }
    
    /**
     * @deprecated use getClassColumnName
     * vrati nazev sloupce ve tvaru $alias.column_name kde column_name odpovida $propertyName ve tride
     * @param string $className
     * @param string $propertyName
     * @param string $alias
     */
    public function getClassCoumnName($className, $propertyName, $alias = NULL) {
        return $this->getClassColumnName($className, $propertyName, $alias);
    }
    
    
    public function getPropertyPDODataType($propertyName) {
        $propertyDescription = $this->classDescription->getPropertyDescription($propertyName);
        return $propertyDescription->getPdoDataType();
    }
    
    public function getClassPropertyPDODataType($className, $propertyName) {
        $classDescription = $this->dBConnection->getClassDesctiption($className);
        $propertyDescription = $classDescription->getPropertyDescription($propertyName);
        return $propertyDescription->getPdoDataType();
    }
    
    /**
     * vrati jmeno pod jakym ma byt zadan parametr v sql dotazu
     * @param integer|string $index na jakem indexu se parameter u find*, update*, *delete jsou indexy cisla u custom* to mohou byt i stringy
     * @return string parametr, ktery se ma vlozit do sql 
     */
    public function getParameterName($index) {
        if( !isset($this->parameters[$index]) ) {
            throw new Exception("Parametr $index není předáván dotazu");
        }
        if( is_array($this->parameters[$index]) ) {
            throw new Exception("Parametr $index nesmí být pole");
        }
        
        $used = 0;
        if( isset($this->repetedParameters[$index]) ) {
            $used = $this->repetedParameters[$index];
        }
        $used = $used + 1;
        $this->repetedParameters[$index] = $used;
        
        return ":arg". $index ."rep". ($used-1);
    }
    
    /**
     * vraci pod jakym aliasem se vraci pocet radku v count dotazu
     * @return string
     */
    function getCountAlias() {
        return $this->countAlias;
    }
    
   /**
    * Vrati expandovana jmena pro vlozeni do IN casti SQL dotazu
    * Parametr pro tento dotaz musi byt pole
    * @param integer|string $index na jakem indexu se parameter
    * @return string parametry, ktere se maji vlozit do sql (bez závorek!)
    */
    public function getInParamersNames($index) {
        if( !isset($this->parameters[$index]) ) {
            throw new Exception("Parametr $index není předáván dotazu");
        }
        if( !is_array($this->parameters[$index]) ) {
            throw new Exception("Pro expandování musí být parametr $index pole");
        }
        
        $used = 0;
        if( isset($this->inExpandedParameters[$index]) ) {
            $used = $this->inExpandedParameters[$index];
        }
        $used = $used + 1;
        $this->inExpandedParameters[$index] = $used;
        
        $paramArray = $this->parameters[$index];
        $paramsExpanded = [];
        foreach($paramArray as $key => $value) {
            $paramsExpanded[] = ":arg". $index ."rep". ($used-1) ."exp". $key;
        }
        
        return implode(", ", $paramsExpanded);
    }
    
    /**
     * @return array vraci pole parametru, kde jsou parametry pod klici pod kterymi se maji predat do statementu
     */
    public function getQueryParameters() {
        $queryParams = [];
        foreach($this->parameters as $index => $value) {
            if( isset($this->repetedParameters[$index]) ) { // je opakovany parametr
                for($i = 0; $i < $this->repetedParameters[$index]; $i++) {
                    $paramIndex = ":arg". $index ."rep". $i;
                    $queryParams[$paramIndex] = $value;
                }
            } else if( isset($this->inExpandedParameters[$index]) ) { // je in expandovany parametr
                for($i = 0; $i < $this->inExpandedParameters[$index]; $i++) {
                    $paramArray = $this->parameters[$index];
                    foreach($paramArray as $key => $inValue) {
                        $paramIndex = ":arg". $index ."rep". $i ."exp". $key;
                        $queryParams[$paramIndex] = $inValue;
                    }
                }
            } else { // ne normalni nepouzity parametr
                $queryParams[":arg". $index] = $value;
            }
        }
        return $queryParams;
    }
    
    /**
     * vrati cast dotazu po group by "$alias.conum_name1, $alias.conum_name2, ... " 
     * @param array $properties list of properties
     * @param boolena $excludeProperties if include or exclude properties specified in $properties
     * @param string $alias
     * @return string
     */
    public function getGroupByProperties($alias = NULL, array $propertieNames = [], $excludeProperties = true) {
        $properties = '';
        if( $excludeProperties ) {
            foreach($this->classDescription->getPropertiesToColumnsTranslation() as $key => $value) {
                $propertyDescription = $this->classDescription->getPropertyDescription($key);
                if( !in_array($propertyDescription->getPropertyName(), $propertieNames) ) {
                    if($properties != '') {
                        $properties .= ', ';
                    }
                    $properties .=  ($alias === NULL ? "" : "$alias.") . $propertyDescription->getColumnName();
                }
            }
        } else {
            foreach($propertieNames as $key) {
                $propertyDescription = $this->classDescription->getPropertyDescription($key);
                if($properties != '') {
                    $properties .= ', ';
                }
                $properties .=  ($alias === NULL ? "" : "$alias.") . $propertyDescription->getColumnName();
            }
        }
        return $properties;
    }
    
    /**
     * when class name is null, uses current domain
     * @param string $alias
     * @param string|null $className
     * @return TableAlias
     */
    public function createTableAlias($alias, $className = null) {
      return new TableAlias($this, $alias, $className);
    }
}